<?php
/**
 * Plugin Name: Path Pilot Pro
 * Plugin URI: https://pathpilot.app
 * Description: Modern WordPress plugin for smart recommendations and analytics.
 * Version: 1.2.0
 * Author: Path Pilot
 * Author URI: https://pathpilot.app
 * Text Domain: path-pilot
 * Requires Plugins: path-pilot
 */
 namespace Path_Pilot;

if (!defined('ABSPATH')) exit;

if (!function_exists('get_plugin_data')) {
    require_once(ABSPATH . 'wp-admin/includes/plugin.php');
}

function get_path_pilot_pro_version() {
    return get_plugin_data(__FILE__)['Version'];
}

add_action('plugins_loaded', function () {

    if (!class_exists(__NAMESPACE__ . '\\Path_Pilot')) {
        add_action('admin_notices', function () {
            echo '<div class="notice notice-error"><p>';
            echo 'Path Pilot Pro requires the Path Pilot plugin to be active and loaded first.';
            echo '</p></div>';
        });
        return;
    }

// Debug: Log when this file is loaded
    Log::info('Path Pilot Pro: Plugin file loaded for URL: ' . ($_SERVER['REQUEST_URI'] ?? 'unknown'));

    define('PATH_PILOT_PRO', true); // Pro version
    define('PATH_PILOT_AI_API_URL', getenv('PATH_PILOT_AI_API_URL') ?: 'https://proxy.pathpilot.app');

    require_once __DIR__ . '/includes/pro/class-path-pilot-admin-pro.php';
    require_once __DIR__ . '/includes/pro/class-path-pilot-ai.php';
    require_once __DIR__ . '/includes/pro/class-path-pilot-chat.php';
    require_once __DIR__ . '/includes/pro/class-path-pilot-embeddings.php';
    require_once __DIR__ . '/includes/pro/parsedown.php';
    require_once __DIR__ . '/includes/pro/class-path-pilot-update-check.php';
    require_once __DIR__ . '/admin/pro/settings-pro.php';

    Log::info("--- Path_Pilot Pro ---");
    Log::info("+++");

    class Path_Pilot_Pro extends Path_Pilot {
        // Use constants from shared class
        const SLUG = Path_Pilot_Shared::SLUG;
        const REST_NAMESPACE = Path_Pilot_Shared::REST_NAMESPACE;

        public function __construct() {
            Log::info('Path Pilot Pro: Main class constructor called');
            // Register pro-only REST endpoint for chat
            add_action('rest_api_init', function() {
                register_rest_route(self::REST_NAMESPACE, '/chat', [
                    'methods' => 'POST',
                    'callback' => [__CLASS__, 'handle_chat'],
                    'permission_callback' => '__return_true',
                ]);
            });

            // Register pro-only REST endpoint for checking credits
            add_action('rest_api_init', function() {
                register_rest_route(self::REST_NAMESPACE, '/credits', [
                    'methods' => 'GET',
                    'callback' => [__CLASS__, 'handle_get_credits'],
                    'permission_callback' => function() {
                        return current_user_can('manage_options');
                    }
                ]);
            });

            // Register pro-only REST endpoint for creating Stripe checkout session
            add_action('rest_api_init', function() {
                register_rest_route(self::REST_NAMESPACE, '/create-topup-checkout', [
                    'methods' => 'POST',
                    'callback' => [__CLASS__, 'handle_create_topup_checkout'],
                    'permission_callback' => function() {
                        return current_user_can('manage_options');
                    }
                ]);
            });

            // Register pro-only REST endpoint for creating Stripe customer portal session
            add_action('rest_api_init', function() {
                register_rest_route(self::REST_NAMESPACE, '/create-portal-session', [
                    'methods' => 'POST',
                    'callback' => [__CLASS__, 'handle_create_portal_session'],
                    'permission_callback' => function() {
                        return current_user_can('manage_options');
                    }
                ]);
            });

            // Register pro-only REST endpoint for testing API keys
            add_action('rest_api_init', function() {
                register_rest_route(self::REST_NAMESPACE, '/test-api-key', [
                    'methods' => 'POST',
                    'callback' => [__CLASS__, 'handle_test_api_key'],
                    'permission_callback' => function() {
                        return current_user_can('manage_options');
                    }
                ]);

                register_rest_route(self::REST_NAMESPACE, '/delete-api-key', [
                    'methods' => 'POST',
                    'callback' => [__CLASS__, 'handle_delete_api_key'],
                    'permission_callback' => function() {
                        return current_user_can('manage_options');
                    }
                ]);
            });

            // Check if scripts are actually printed
            add_action('wp_print_scripts', function() {
                global $wp_scripts;
                if (isset($wp_scripts->queue) && in_array('path-pilot-interactivity', $wp_scripts->queue)) {
                    Log::info('Path Pilot: index.js script is in the print queue');
                } else {
                    Log::info('Path Pilot: index.js script is NOT in the print queue');
                }
                if (isset($wp_scripts->queue) && in_array('path-pilot-tracking', $wp_scripts->queue)) {
                    Log::info('Path Pilot: tracking.js script is in the print queue');
                } else {
                    Log::info('Path Pilot: tracking.js script is NOT in the print queue');
                }
            });

            // Scheduled path analysis
            add_action('path_pilot_analyze_paths', [__NAMESPACE__ . '\\Path_Pilot_Shared', 'analyze_paths']);
            if (!wp_next_scheduled('path_pilot_analyze_paths')) {
                wp_schedule_event(time(), 'hourly', 'path_pilot_analyze_paths');
            }

            // --- Daily snapshot cron ---
            add_action('path_pilot_daily_snapshot', [__NAMESPACE__ . '\\Path_Pilot_Shared', 'run_daily_snapshot']);
            if (!wp_next_scheduled('path_pilot_daily_snapshot')) {
                wp_schedule_event(strtotime('tomorrow'), 'daily', 'path_pilot_daily_snapshot');
            }
            // --- End daily snapshot cron ---

            // Hook: Generate embedding on post publish/update
            add_action('save_post', [__CLASS__, 'maybe_generate_embedding'], 10, 2);

            new Path_Pilot_Admin_Pro();
            // Path_Pilot_Settings_Pro now handled by Path_Pilot_Admin_Pro

            new Path_Pilot_Chat();
            new Path_Pilot_AI();

            // Enqueue Pro frontend scripts
            add_action('wp_enqueue_scripts', [$this, 'enqueue_pro_scripts']);
        }

        /**
         * Enqueue Pro frontend scripts and styles
         */
        public function enqueue_pro_scripts() {
            // Only enqueue if Pro is active and chat is enabled
            if (!self::get_api_key_status()['valid']) {
                return;
            }

            // Enqueue the Pro chat CSS
            wp_enqueue_style(
                'path-pilot-pro-chat',
                plugins_url('assets/css/chat.css', __FILE__),
                [],
                get_path_pilot_pro_version()
            );

            // Enqueue the Pro chat script
            wp_enqueue_script(
                'path-pilot-pro-chat',
                plugins_url('scripts/chat.js', __FILE__),
                ['path-pilot-interactivity'], // Depends on the main interactivity script
                get_path_pilot_pro_version(),
                false // Load in header
            );

            Log::info('Path Pilot Pro: Enqueued chat.js script and chat.css styles');
        }

        // --- Pro-only REST handler for chat ---
        public static function handle_chat($request) {
            $params = $request->get_json_params();

            // Debug logging to see what we receive
            Log::info('PATH_PILOT CHAT DEBUG: Received params: ' . print_r($params, true));

            $question = trim(sanitize_text_field($params['question'] ?? ''));
            $has_seen_recommendation = !empty($params['has_seen_recommendation']);
            $history = $params['history'] ?? [];

            Log::info('PATH_PILOT CHAT DEBUG: Extracted question: "' . $question . '"');
            Log::info('PATH_PILOT CHAT DEBUG: has_seen_recommendation: ' . ($has_seen_recommendation ? 'true' : 'false'));
            Log::info('PATH_PILOT CHAT DEBUG: History count: ' . count($history));

            if (!$has_seen_recommendation) {
                return [ 'error' => 'Chat is only available after seeing a recommendation.' ];
            }

            if (!$question) {
                return [ 'error' => 'No question provided.' ];
            }

            $api_key = get_option('path_pilot_api_key', '');

            // Check if API key is valid first
            if (empty($api_key) || !self::is_api_key_valid($api_key)) {
                return ['error' => 'Chat feature requires a valid Path Pilot Pro license key. Please visit the settings page to configure your API key.'];
            }

            // Check if we're on localhost for development
            $current_domain = parse_url(home_url(), PHP_URL_HOST);
            $is_localhost = (strpos($current_domain, 'localhost') !== false || strpos($current_domain, '.local') !== false);

            // Check if user has available credits before processing chat (skip for localhost with valid subscription)
            if ($api_key && !$is_localhost) {
                $credit_data = self::get_all_credit_balances($api_key);
                if (!is_wp_error($credit_data)) {
                    $total_remaining = $credit_data['total_remaining'] ?? 0;
                    if ($total_remaining <= 0) {
                        return [ 'error' => 'I\'m temporarily unable to respond at the moment. Please try again later!' ];
                    }
                }
            } elseif ($is_localhost) {
                // On localhost, just check if they have a valid subscription, don't consume credits
                $subscription_status = self::get_subscription_status($api_key);
                if (!$subscription_status['has_active_subscription']) {
                    return [ 'error' => 'I\'m temporarily unable to respond at the moment. Please try again later!' ];
                }
                Log::info('Path Pilot: Chat allowed on localhost for valid subscriber - no credits consumed');
            }

            $search_query = $question;

            // Rewrite the question using OpenAI for better semantic search
            $rewrite_reply = Path_Pilot_AI::rewrite_query_with_openai($question, $api_key);
            if ($rewrite_reply) {
                $search_query = $rewrite_reply;
                Log::info('Path Pilot Chat: Rewrote query from "' . $question . '" to "' . $search_query . '"');
            }

            // Get embedding for the search query
            $query_embedding = Path_Pilot_AI::get_openai_embedding($search_query, $api_key);
            if (!$query_embedding) {
                return [ 'error' => 'Could not generate embedding.' ];
            }

            // Find relevant pages using semantic search
            global $wpdb;
            $rows = $wpdb->get_results("SELECT post_id, embedding FROM {$wpdb->prefix}path_pilot_vectors");
            $scores = [];

            // Array of goal page post IDs
            $goal_pages = Path_Pilot_Shared::get_goal_pages();
            // Array of conversion page post IDs
            $conversion_pages = Path_Pilot_Shared::get_conversion_pages();
            // Array of allowed content types
            $allowed_content_types = Path_Pilot_Shared::get_allowed_content_types();

            Log::info('Path Pilot Chat: Using content types for search: ' . implode(', ', $allowed_content_types));

            foreach ($rows as $row) {
                if (in_array($row->post_id, $conversion_pages)) continue; // Skip only conversion pages, allow goal pages

                // Verify the post is published and publicly accessible
                $post = get_post($row->post_id);
                if (!$post || $post->post_status !== 'publish' || post_password_required($post)) continue;

                // Check if post type is allowed for recommendations
                if (!in_array($post->post_type, $allowed_content_types)) {
                    Log::info("Path Pilot Chat: Skipping post '{$post->post_title}' - type '{$post->post_type}' not in allowed types");
                    continue;
                }

                $embedding = maybe_unserialize($row->embedding);
                if (!is_array($embedding)) continue;

                // Base semantic similarity score (0-1)
                $semantic_score = self::cosine_similarity($query_embedding, $embedding);

                // Keyword/title matching boost
                $keyword_boost = self::calculate_keyword_boost($question, $search_query, $post);

                // Check for exact title matches and boost them significantly
                $title_lower = strtolower(trim($post->post_title));
                $query_lower = strtolower(trim($question));
                $exact_title_boost = 0;

                if ($title_lower === $query_lower) {
                    $exact_title_boost = 0.5; // Huge boost for exact matches
                } elseif (strpos($title_lower, $query_lower) !== false || strpos($query_lower, $title_lower) !== false) {
                    $exact_title_boost = 0.3; // Good boost for partial matches
                }

                // Combine scores: 60% semantic + 25% keyword matching + 15% title boost
                $final_score = ($semantic_score * 0.6) + ($keyword_boost * 0.25) + $exact_title_boost;

                // Debug logging for search relevance
                Log::info("Path Pilot Search: Post '{$post->post_title}' - Semantic: " . round($semantic_score, 3) . ", Keyword: " . round($keyword_boost, 3) . ", Title: " . round($exact_title_boost, 3) . ", Final: " . round($final_score, 3));

                $scores[$row->post_id] = $final_score;
            }

            arsort($scores);
            $top_ids = array_slice(array_keys($scores), 0, 5, true);

            // Debug: Log final ranking
            Log::info("Path Pilot Search: Final ranking for query '$question':");
            $rank = 1;
            foreach ($top_ids as $post_id) {
                $post = get_post($post_id);
                if ($post) {
                    Log::info("  $rank. '{$post->post_title}' (score: " . round($scores[$post_id], 3) . ")");
                    $rank++;
                }
            }

            $context = [];

            foreach ($top_ids as $post_id) {
                $post = get_post($post_id);
                // Only include published posts that are publicly accessible
                if ($post && $post->post_status === 'publish' && !post_password_required($post)) {
                    $context[] = [
                        'title' => $post->post_title,
                        'excerpt' => wp_trim_words($post->post_content, 300),
                        'url' => parse_url(get_permalink($post->ID), PHP_URL_PATH)
                    ];
                }
            }

            // Build system message with site context
            $system_message = "You are a helpful assistant for a website. Here are some relevant pages from the site:";
            foreach ($context as $c) {
                $system_message .= "\n- " . $c['title'] . ": " . $c['excerpt'] . " (URL: " . $c['url'] . ")";
            }
            // Empirically, the "Take a deep breath. Slow down." message reduces the number of mismatched URLs.
            $system_message .= "\nIMPORTANT: Only recommend pages from the list above. Do NOT invent or mention any pages or URLs that are not in the list.
            If none are relevant, say so, but select at least one page so the user has something to click.
            Make your answer, well formed HTML. When you recommend a page use an A tag. Put your page recommendations in LI tags within a UL element.
            Only use titles and URLs from the list provided.
            Take a deep breath. Slow down.
            When suggesting a URL, double check that the URL your are suggesting matches the URL in the list.
            Before inserting a URL into the HTML response, verify that the number of characters in the URL is the same number as the number of characters in the corresponding URL from the list above.
            Do not recommend pages on other websites.";

            // Build messages array: system message + history + current user question
            $messages = [ [ 'role' => 'system', 'content' => $system_message ] ];
            if (!empty($history)) {
                foreach ($history as $msg) {
                    if (isset($msg['role'], $msg['content']) && in_array($msg['role'], ['user', 'assistant'])) {
                        $messages[] = [
                            'role' => $msg['role'],
                            'content' => $msg['content']
                        ];
                    }
                }
            }
            // Ensure the current user question is the last message
            $last = end($messages);
            if (!($last && $last['role'] === 'user' && $last['content'] === $question)) {
                $messages[] = [ 'role' => 'user', 'content' => $question ];
            }
            // Log the system message and last user message for debugging
            Log::info('Path Pilot: System/context message: ' . $system_message);
            Log::info('Path Pilot: Last user message: ' . $question);
            $reply = Path_Pilot_AI::chat_with_openai(['messages' => $messages], $api_key);
            if (!$reply) {
                return [ 'error' => 'Sorry, something went wrong, can you ask again?' ];
            }

            $html = $reply;

            // Optionally, restrict links to only allowed URLs as before
            $allowed = [];
            foreach ($context as $c) {
                $allowed[$c['title']] = $c['url'];
            }
            $html = preg_replace_callback('/<a[^>]+href=["\']([^"\']+)["\'][^>]*>([^<]+)<\/a>/', function($m) use ($allowed) {
                $url = $m[1];
                $title = $m[2];
                if (in_array($url, $allowed)) {
                    return '<a href="' . esc_attr($url) . '" target="_self">' . esc_html($title) . '</a>';
                } else {
                    Log::info('Path Pilot: Link not allowed: ' . $url);
                    return esc_html($title);
                }
            }, $html);

            return [ 'reply' => $html, 'context' => $context ];
        }

        /**
         * REST API handler to get the current credit balance.
         * Accessible via GET /wp-json/path-pilot/v1/credits
         *
         * @param WP_REST_Request $request
         * @return WP_REST_Response
         */
        public static function handle_get_credits($request) {
            $api_key = get_option('path_pilot_api_key', '');
            if (empty($api_key)) {
                return new \WP_REST_Response(['error' => 'API key is not configured.'], 400);
            }

            // Check if we're on localhost - if so, allow chat for valid subscribers regardless of credits
            $current_domain = parse_url(home_url(), PHP_URL_HOST);
            $is_localhost = (strpos($current_domain, 'localhost') !== false || strpos($current_domain, '.local') !== false);

            // Get both monthly and anytime credit balances
            $credit_data = self::get_all_credit_balances($api_key);

            if (is_wp_error($credit_data)) {
                return new \WP_REST_Response(['error' => $credit_data->get_error_message()], 500);
            }

            // If on localhost and has valid subscription, simulate having credits for chat functionality
            if ($is_localhost && isset($credit_data['has_active_subscription']) && $credit_data['has_active_subscription']) {
                // Override total_remaining to allow chat on localhost for valid subscribers
                $credit_data['total_remaining'] = max(1, $credit_data['total_remaining']); // Ensure at least 1 credit for chat to work
                $credit_data['localhost_override'] = true; // Flag to indicate this is a localhost override
            }

            return new \WP_REST_Response($credit_data, 200);
        }

        /**
         * REST API handler to create Stripe checkout session for credit top-up
         * Accessible via POST /wp-json/path-pilot/v1/create-topup-checkout
         *
         * @param WP_REST_Request $request
         * @return WP_REST_Response
         */
        public static function handle_create_topup_checkout($request) {
            $params = $request->get_json_params();
            $credits = intval($params['credits'] ?? 0);
            $subscription_id = sanitize_text_field($params['subscription_id'] ?? '');

            if (!$credits || !in_array($credits, [5000, 10000, 25000])) {
                return new \WP_REST_Response(['error' => 'Invalid credit amount'], 400);
            }

            if (!$subscription_id) {
                return new \WP_REST_Response(['error' => 'Subscription ID required'], 400);
            }

            // Map credits to price IDs (you'll need to update these with your actual price IDs)
            $price_map = [
                5000 => 'price_1S3HxoLgrqsGdSsVQYkcBm2D',   // Replace with actual price ID
                10000 => 'price_1S3HxrLgrqsGdSsVQo482pTK',  // Replace with actual price ID
                25000 => 'price_1S3HxtLgrqsGdSsVzxrsnZM0'   // Replace with actual price ID
            ];

            if (!isset($price_map[$credits])) {
                return new \WP_REST_Response(['error' => 'Price not found for credit amount'], 400);
            }

            try {
                // Create Stripe checkout session
                $checkout_session = \Stripe\Checkout\Session::create([
                    'payment_method_types' => ['card'],
                    'line_items' => [[
                        'price' => $price_map[$credits],
                        'quantity' => 1,
                    ]],
                    'mode' => 'payment', // One-time payment
                    'success_url' => home_url('/wp-admin/admin.php?page=path-pilot-settings&tab=pro&topup=success'),
                    'cancel_url' => home_url('/wp-admin/admin.php?page=path-pilot-settings&tab=pro&topup=cancelled'),
                    'metadata' => [
                        'type' => 'credit_topup',
                        'credits' => $credits,
                        'subscription_id' => $subscription_id, // Critical: link to subscription
                        'installation_domain' => parse_url(home_url(), PHP_URL_HOST)
                    ]
                ]);

                return new \WP_REST_Response(['checkout_url' => $checkout_session->url], 200);

            } catch (\Exception $e) {
                return new \WP_REST_Response(['error' => 'Failed to create checkout session: ' . $e->getMessage()], 500);
            }
        }

        /**
         * REST API handler to create a Stripe Billing Portal session
         * Accessible via POST /wp-json/path-pilot/v1/create-portal-session
         *
         * @param WP_REST_Request $request
         * @return WP_REST_Response
         */
        public static function handle_create_portal_session($request) {
            // Get Stripe customer ID from license key metadata
            $api_key = get_option('path_pilot_api_key', '');
            if (empty($api_key)) {
                return new \WP_REST_Response(['error' => 'No license key found.'], 400);
            }

            $customer_id = self::get_stripe_customer_id($api_key);
            if (empty($customer_id)) {
                return new \WP_REST_Response(['error' => 'Stripe customer not found for this installation.'], 400);
            }

            // Call external stripe-hook service to create a portal session
            // Use env var if provided; otherwise default to the known App Runner URL
            $hook_base = getenv('STRIPE_HOOK_BASE_URL') ?: 'https://wm2s3greew.us-east-1.awsapprunner.com';
            $endpoint = rtrim($hook_base, '/') . '/portal/session';

            $payload = [
                'customer_id' => $customer_id,
                'return_url' => admin_url('admin.php?page=path-pilot-settings&tab=pro')
            ];

            // Debug log to verify payload being sent to billing service
            Log::info('Path Pilot: Creating Stripe portal session via ' . $endpoint . ' for customer ' . $customer_id . ' with return_url ' . $payload['return_url']);

            $args = [
                'headers' => [
                    'Content-Type' => 'application/json',
                ],
                'body' => wp_json_encode($payload),
                'timeout' => 15,
                'sslverify' => (strpos(home_url(), 'localhost') !== false || strpos(home_url(), '.local') !== false),
            ];

            $response = wp_remote_post($endpoint, $args);
            if (is_wp_error($response)) {
                Log::info('Path Pilot: Portal session error contacting billing service: ' . $response->get_error_message());
                return new \WP_REST_Response(['error' => 'Failed to contact billing service: ' . $response->get_error_message()], 500);
            }
            $code = wp_remote_retrieve_response_code($response);
            $raw_body = wp_remote_retrieve_body($response);
            $body = json_decode($raw_body, true);
            Log::info('Path Pilot: Billing service response code ' . $code . ' body: ' . $raw_body);
            if ($code !== 200 || !isset($body['url'])) {
                $msg = isset($body['detail']) ? $body['detail'] : (isset($body['error']) ? $body['error'] : 'Unexpected response from billing service');
                $resp = ['error' => $msg];
                if (isset($body['fallback_url'])) {
                    $resp['fallback_url'] = $body['fallback_url'];
                }
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    $resp['debug'] = [
                        'status_code' => $code,
                        'endpoint' => $endpoint,
                        'customer_id' => $customer_id,
                        'response' => $raw_body
                    ];
                }
                return new \WP_REST_Response($resp, 500);
            }

            return new \WP_REST_Response(['url' => $body['url']], 200);
        }

        /**
         * Retrieves the Stripe customer ID from the key metadata (LiteLLM /key/info)
         *
         * @param string $monthly_api_key The monthly LiteLLM API key
         * @return string Stripe customer ID or empty string
         */
        public static function get_stripe_customer_id($monthly_api_key) {
            $url = PATH_PILOT_AI_API_URL . '/key/info';

            $args = [
                'headers' => [
                    'Authorization' => 'Bearer ' . $monthly_api_key,
                ],
                'timeout' => 15,
                'sslverify' => (strpos(home_url(), 'localhost') !== false || strpos(home_url(), '.local') !== false),
            ];

            $response = wp_remote_get($url, $args);
            if (is_wp_error($response)) {
                return '';
            }
            $response_code = wp_remote_retrieve_response_code($response);
            if ($response_code !== 200) {
                return '';
            }
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);

            if (isset($data['info']['metadata']['user_id']) && !empty($data['info']['metadata']['user_id'])) {
                // In our system, metadata.user_id stores the Stripe customer ID
                return (string) $data['info']['metadata']['user_id'];
            }
            return '';
        }

        // Removed local Stripe secret usage; portal sessions are created by external service

        /**
         * REST API handler to test an API key without saving it
         * Accessible via POST /wp-json/path-pilot/v1/test-api-key
         *
         * @param WP_REST_Request $request
         * @return WP_REST_Response
         */
        public static function handle_test_api_key($request) {
            $params = $request->get_json_params();
            $api_key = sanitize_text_field($params['api_key'] ?? '');

            if (empty($api_key)) {
                return new \WP_REST_Response(['error' => 'No API key provided'], 400);
            }

            // Test the API key by attempting to get credit balance
            $credit_data = self::get_credit_balance($api_key);

            if (is_wp_error($credit_data)) {
                return new \WP_REST_Response([
                    'valid' => false,
                    'error' => $credit_data->get_error_message()
                ], 200); // Return 200 so frontend can handle the error gracefully
            }

            // Get subscription status as well
            $subscription_status = self::get_subscription_status($api_key);

            return new \WP_REST_Response([
                'valid' => true,
                'message' => 'API key is valid and active',
                'credits' => $credit_data,
                'subscription' => $subscription_status
            ], 200);
        }

        public static function handle_delete_api_key($request) {
            // Delete the API key option
            delete_option('path_pilot_api_key');

            return new \WP_REST_Response(['success' => true, 'message' => 'API key deleted'], 200);
        }

        /**
         * Retrieves both monthly and anytime credit balances.
         *
         * @param string $monthly_api_key The monthly LiteLLM API key.
         * @return array|WP_Error An array with credit information, or a WP_Error on failure.
         */
        public static function get_all_credit_balances($monthly_api_key) {
            $cost_per_credit = 0.001; // 1 credit = $0.001

            // Get monthly credits
            $monthly_balance = self::get_credit_balance($monthly_api_key);
            if (is_wp_error($monthly_balance)) {
                return $monthly_balance;
            }

            // Check subscription status by getting monthly key info
            $subscription_status = self::get_subscription_status($monthly_api_key);

            // Get anytime key from monthly key metadata
            $anytime_key = self::get_anytime_key_from_monthly($monthly_api_key);
            $anytime_balance = ['remaining' => 0, 'total' => 0];

            if ($anytime_key) {
                $anytime_balance = self::get_credit_balance($anytime_key);
                if (is_wp_error($anytime_balance)) {
                    // If anytime key fails, just set to 0 and continue
                    $anytime_balance = ['remaining' => 0, 'total' => 0];
                }
            }

            return [
                'monthly' => $monthly_balance,
                'anytime' => $anytime_balance,
                'total_remaining' => $monthly_balance['remaining'] + $anytime_balance['remaining'],
                'total_credits' => $monthly_balance['total'] + $anytime_balance['total'],
                'has_active_subscription' => $subscription_status['has_active_subscription'],
                'subscription_id' => $subscription_status['subscription_id']
            ];
        }

        /**
         * Gets the anytime key from monthly key metadata.
         *
         * @param string $monthly_api_key The monthly LiteLLM API key.
         * @return string|null The anytime key or null if not found.
         */
        public static function get_anytime_key_from_monthly($monthly_api_key) {
            $url = PATH_PILOT_AI_API_URL . '/key/info';

            $args = [
                'headers' => [
                    'Authorization' => 'Bearer ' . $monthly_api_key,
                ],
                'timeout' => 15,
                'sslverify' => (strpos(home_url(), 'localhost') !== false || strpos(home_url(), '.local') !== false),
            ];

            $response = wp_remote_get($url, $args);

            if (is_wp_error($response)) {
                return null;
            }

            $response_code = wp_remote_retrieve_response_code($response);
            if ($response_code !== 200) {
                return null;
            }

            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);

            if (isset($data['info']['metadata']['anytime_key'])) {
                return $data['info']['metadata']['anytime_key'];
            }

            return null;
        }

        /**
         * Retrieves the remaining and total credit balance for a given LiteLLM API key.
         *
         * @param string $api_key The user's LiteLLM API key.
         * @return array|WP_Error An array with 'remaining' and 'total' credits, or a WP_Error on failure.
         */
        public static function get_credit_balance($api_key) {
            $url = PATH_PILOT_AI_API_URL . '/key/info';
            $cost_per_credit = 0.001; // 1 credit = $0.001

            $args = [
                'headers' => [
                    'Authorization' => 'Bearer ' . $api_key,
                ],
                'timeout' => 15,
                /**
                 * FOR LOCAL DEVELOPMENT ONLY:
                 * Disable SSL verification if you get cURL timeout errors. This is often necessary
                 * in local environments that don't have up-to-date CA root certificates.
                 * This line should be removed for production.
                 */
                'sslverify' => (strpos(home_url(), 'localhost') !== false || strpos(home_url(), '.local') !== false),
            ];

            $response = wp_remote_get($url, $args);

            if (is_wp_error($response)) {
                return $response; // Return the WP_Error object
            }

            $response_code = wp_remote_retrieve_response_code($response);
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);

            if ($response_code !== 200) {
                $error_message = isset($data['detail']) ? $data['detail'] : 'Failed to retrieve balance from API.';
                return new \WP_Error('api_error', $error_message, ['status' => $response_code]);
            }

            $remaining_credits = 0;
            $total_credits = 0;

            // Check if the response has the expected nested structure
            if (isset($data['info']['max_budget']) && isset($data['info']['spend'])) {
                $max_budget = (float) $data['info']['max_budget'];
                $spend = (float) $data['info']['spend'];
                $remaining_budget = $max_budget - $spend;

                $total_credits = floor($max_budget / $cost_per_credit);
                $remaining_credits = floor($remaining_budget / $cost_per_credit);
            }

            return [
                'remaining' => (int) $remaining_credits,
                'total'     => (int) $total_credits,
            ];
        }

        /**
         * Gets subscription status from monthly key metadata.
         *
         * @param string $monthly_api_key The monthly LiteLLM API key.
         * @return array Array with has_active_subscription boolean and subscription_id string.
         */
        public static function get_subscription_status($monthly_api_key) {
            $url = PATH_PILOT_AI_API_URL . '/key/info';

            $args = [
                'headers' => [
                    'Authorization' => 'Bearer ' . $monthly_api_key,
                ],
                'timeout' => 15,
                'sslverify' => (strpos(home_url(), 'localhost') !== false || strpos(home_url(), '.local') !== false),
            ];

            $response = wp_remote_get($url, $args);

            if (is_wp_error($response)) {
                return ['has_active_subscription' => false, 'subscription_id' => ''];
            }

            $response_code = wp_remote_retrieve_response_code($response);
            if ($response_code !== 200) {
                return ['has_active_subscription' => false, 'subscription_id' => ''];
            }

            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);

            // Check if key is blocked (subscription cancelled/expired)
            $is_blocked = isset($data['info']['blocked']) && $data['info']['blocked'] === true;

            // Check if key has budget > 0 (active subscription)
            $max_budget = isset($data['info']['max_budget']) ? (float) $data['info']['max_budget'] : 0;

            // Get subscription ID from metadata
            $subscription_id = isset($data['info']['metadata']['subscription_id']) ? $data['info']['metadata']['subscription_id'] : '';

            // Has active subscription if not blocked, has budget, and has subscription ID
            $has_active_subscription = !$is_blocked && $max_budget > 0 && !empty($subscription_id);

            return [
                'has_active_subscription' => $has_active_subscription,
                'subscription_id' => $subscription_id
            ];
        }

        public static function get_api_key() {
            return get_option('path_pilot_api_key', '');
        }

        /**
         * Validates if the stored API key is working and active.
         *
         * @param string|null $api_key Optional API key to test. If null, uses stored option.
         * @return bool True if API key is valid and working, false otherwise.
         */
        public static function is_api_key_valid($api_key = null) {
            if ($api_key === null) {
                $api_key = get_option('path_pilot_api_key', '');
            }

            // Empty API key is invalid
            if (empty($api_key)) {
                return false;
            }

            // Test the API key by attempting to get credit balance
            $credit_data = self::get_credit_balance($api_key);

            // If we get a WP_Error, the API key is invalid
            if (is_wp_error($credit_data)) {
                return false;
            }

            // If we get valid data structure, the API key works
            return isset($credit_data['remaining']) && isset($credit_data['total']);
        }

        /**
         * Gets the validation status and error message for the current API key.
         *
         * @return array Array with 'valid' boolean and 'message' string.
         */
        public static function get_api_key_status($api_key_to_test = null) {
            if ($api_key_to_test === null) {
                $api_key_to_test = get_option('path_pilot_api_key', '');
            }

            if (empty($api_key_to_test)) {
                return [
                    'valid' => false,
                    'message' => 'No API key configured. Please enter your Path Pilot Pro license key.'
                ];
            }

            $credit_data = self::get_credit_balance($api_key_to_test);

            if (is_wp_error($credit_data)) {
                $error_message = $credit_data->get_error_message();
                return [
                    'valid' => false,
                    'message' => 'API key validation failed: ' . $error_message
                ];
            }

            return [
                'valid' => true,
                'message' => 'API key is valid and active.'
            ];
        }

        // --- Pro-only methods for embeddings and AI features ---

        // Generate and store embedding for a post
        public static function maybe_generate_embedding($post_id, $post) {
            global $wpdb;

            // Check if post type is allowed for embeddings
            $allowed_content_types = Path_Pilot_Shared::get_allowed_content_types();

            if (in_array($post->post_type, $allowed_content_types)) {
                if ($post->post_status === 'publish' && !post_password_required($post)) {
                    // Generate embedding for published, non-password-protected posts
                    $api_key = get_option('path_pilot_api_key', '');
                    if (!$api_key) return false;

                    Log::info("Path Pilot Embeddings: Generating embedding for {$post->post_type}: '{$post->post_title}'");

                    $text = $post->post_title . "\n\n" . $post->post_title . " - " . $post->post_content; // Repeat title for emphasis
                    $embedding = Path_Pilot_AI::get_openai_embedding($text, $api_key);

                    if ($embedding) {
                        $wpdb->replace($wpdb->prefix . 'path_pilot_vectors', [
                            'post_id' => $post_id,
                            'embedding' => maybe_serialize($embedding),
                            'updated_at' => current_time('mysql')
                        ]);
                        return true;
                    }
                } else {
                    // Remove embedding if post is no longer published or is password protected
                    $wpdb->delete($wpdb->prefix . 'path_pilot_vectors', ['post_id' => $post_id]);
                }
            }
            return false;
        }

        // Calculate keyword/title matching boost for search relevance
        public static function calculate_keyword_boost($original_query, $search_query, $post) {
            $boost = 0.0;

            // Normalize text for comparison
            $title = strtolower(trim($post->post_title));
            $content = strtolower(trim(wp_strip_all_tags($post->post_content)));
            $original_lower = strtolower(trim($original_query));
            $search_lower = strtolower(trim($search_query));

            // Extract keywords from queries (remove common words)
            $stop_words = ['the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for', 'of', 'with', 'by', 'about', 'i', 'me', 'my', 'we', 'us', 'you', 'your', 'he', 'she', 'it', 'they', 'them', 'what', 'where', 'when', 'why', 'how', 'is', 'are', 'was', 'were', 'be', 'been', 'have', 'has', 'had', 'do', 'does', 'did', 'will', 'would', 'could', 'should', 'can', 'may', 'might'];

            $original_keywords = array_filter(explode(' ', $original_lower), function($word) use ($stop_words) {
                return strlen($word) > 2 && !in_array($word, $stop_words);
            });

            $search_keywords = array_filter(explode(' ', $search_lower), function($word) use ($stop_words) {
                return strlen($word) > 2 && !in_array($word, $stop_words);
            });

            // Combine keywords from both queries
            $all_keywords = array_unique(array_merge($original_keywords, $search_keywords));

            foreach ($all_keywords as $keyword) {
                // Exact title match gets highest boost
                if ($title === $keyword) {
                    $boost += 1.0;  // Perfect match
                    continue;
                }

                // Title contains keyword gets high boost
                if (strpos($title, $keyword) !== false) {
                    $boost += 0.8;
                    continue;
                }

                // Title similar to keyword (partial match)
                if (self::string_similarity($title, $keyword) > 0.7) {
                    $boost += 0.6;
                    continue;
                }

                // Content contains keyword gets moderate boost
                if (strpos($content, $keyword) !== false) {
                    $boost += 0.2;
                }
            }

            // Normalize boost to 0-1 range (avoid division by zero)
            $keyword_count = count($all_keywords);
            return $keyword_count > 0 ? min(1.0, $boost / $keyword_count) : 0.0;
        }

        // Calculate string similarity (simple Levenshtein-based)
        public static function string_similarity($str1, $str2) {
            $len1 = strlen($str1);
            $len2 = strlen($str2);
            if ($len1 == 0) return $len2 == 0 ? 1 : 0;
            if ($len2 == 0) return 0;

            $distance = levenshtein($str1, $str2);
            $max_len = max($len1, $len2);
            return 1 - ($distance / $max_len);
        }

        // Cosine similarity between two vectors (used for semantic search)
        public static function cosine_similarity($a, $b) {
            $dot = 0.0;
            $norm_a = 0.0;
            $norm_b = 0.0;
            $len = min(count($a), count($b));

            for ($i = 0; $i < $len; $i++) {
                $dot += $a[$i] * $b[$i];
                $norm_a += $a[$i] * $a[$i];
                $norm_b += $b[$i] * $b[$i];
            }

            if ($norm_a == 0 || $norm_b == 0) return 0.0;
            return $dot / (sqrt($norm_a) * sqrt($norm_b));
        }




    }

    // Initialize plugin
    new Path_Pilot_Pro();
    new Path_Pilot_Update_Check(__FILE__);
});
