<?php
namespace Path_Pilot;

// Path Pilot Embeddings logic
if (!defined('ABSPATH')) exit;

class Path_Pilot_Embeddings {
    // Get OpenAI embedding for a text string
    public static function get_openai_embedding($text, $api_key, $max_chars = 16000, $min_chars = 2000) {
        // Track API call
        $calls = (int) get_option('path_pilot_openai_api_calls', 0);
        update_option('path_pilot_openai_api_calls', $calls + 1);
        update_option('path_pilot_openai_api_last', current_time('mysql'));

        // Truncate to max_chars
        if (mb_strlen($text) > $max_chars) {
            $text = mb_substr($text, 0, $max_chars);
        }

        $body = [
            'input' => $text,
            'model' => 'text-embedding-3-small'
        ];
        $response = wp_remote_post(PATH_PILOT_AI_API_URL . '/embeddings', [
            'headers' => [
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json'
            ],
            'body' => json_encode($body),
            'timeout' => 30
        ]);
        if (is_wp_error($response)) {
            update_option('path_pilot_openai_api_error', $response->get_error_message());
            Log::info('Path Pilot: OpenAI WP_Error: ' . $response->get_error_message());
            return ['success' => false, 'error' => $response->get_error_message()];
        }
        $body_str = wp_remote_retrieve_body($response);
        $data = json_decode($body_str, true);
        if (isset($data['data'][0]['embedding'])) {
            update_option('path_pilot_openai_api_error', '');
            return ['success' => true, 'embedding' => $data['data'][0]['embedding']];
        }
        if (isset($data['error']['message'])) {
            $error_msg = $data['error']['message'];
            update_option('path_pilot_openai_api_error', $error_msg);
            Log::info('Path Pilot: OpenAI API error: ' . $error_msg);

            // Check for context length error and retry with a smaller chunk
            if (
                strpos($error_msg, 'maximum context length') !== false ||
                strpos($error_msg, 'tokens') !== false
            ) {
                // Reduce by 25% and retry, but not below $min_chars
                $new_max = intval($max_chars * 0.9);
                if ($new_max >= $min_chars) {
                    Log::info('Path Pilot: Retrying embedding with reduced length: ' . $new_max . ' chars');
                    return self::get_openai_embedding($text, $api_key, $new_max, $min_chars);
                }
            }
            return ['success' => false, 'error' => $error_msg];
        } else {
            Log::info('Path Pilot: OpenAI API unknown error. Response: ' . $body_str);
            return ['success' => false, 'error' => 'Unknown error from OpenAI API.'];
        }
    }

    // Cosine similarity between two embedding arrays
    public static function cosine_similarity($a, $b) {
        $dot = 0.0; $normA = 0.0; $normB = 0.0;
        $len = min(count($a), count($b));
        for ($i = 0; $i < $len; $i++) {
            $dot += $a[$i] * $b[$i];
            $normA += $a[$i] * $a[$i];
            $normB += $b[$i] * $b[$i];
        }
        if ($normA == 0 || $normB == 0) return 0.0;
        return $dot / (sqrt($normA) * sqrt($normB));
    }
}
