<?php
namespace Path_Pilot;

// Path Pilot Admin functionality
if (!defined('ABSPATH')) exit;

class Path_Pilot_Admin_Pro extends Path_Pilot_Admin {
    public function __construct() {
        parent::__construct(); // Call parent constructor to ensure base hooks are registered

        // Add admin notices for Path Pilot pages
        add_action('admin_notices', [$this, 'show_pro_setup_notice']);
        add_action('path_pilot_show_pro_status_message', [$this, 'show_pro_status_message']);
        add_action('path_pilot_render_embedding_coverage', [$this, 'render_embedding_coverage']);

        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_js']);
        add_action('wp_ajax_get_pages_needing_embeddings', [$this, 'ajax_get_pages_needing_embeddings']);
        add_action('wp_ajax_recompute_embeddings_batch', [$this, 'ajax_recompute_embeddings_batch']);

        // Background full embeddings refresh when license is activated/changed
        add_action('path_pilot_recompute_embeddings_full', [$this, 'recompute_all_embeddings_background']);
        // Unified scheduler: allows UI (manual) and server (auto) to trigger same job
        add_action('wp_ajax_path_pilot_schedule_embeddings_refresh', [$this, 'ajax_schedule_embeddings_refresh']);

        // Hook into settings save to schedule embeddings refresh when license changes
        add_action('admin_post_path_pilot_save_settings', [$this, 'handle_license_key_change'], 5);

        // The Pro plugin registers its own menu entirely when active
        add_action('admin_menu', [$this, 'add_pro_admin_menu_pages']);
    }

    /**
     * Add all admin menu pages for the Pro version.
     */
    public function add_pro_admin_menu_pages() {
        error_log('Path Pilot Pro Admin: add_pro_admin_menu_pages called.');
        // Add Path Pilot main menu
        add_menu_page(
            'Path Pilot',
            'Path Pilot',
            'manage_options',
            'path-pilot',
            [$this, 'render_home_page'],
            'dashicons-airplane',
            25
        );

        // Add submenu page for Home
        add_submenu_page(
            'path-pilot', // Parent slug
            'Path Pilot Home',
            'Home',
            'manage_options',
            'path-pilot', // Menu slug (same as parent for main home page)
            [$this, 'render_home_page']
        );

        // Add submenu page for Analytics
        add_submenu_page(
            'path-pilot',
            'Path Pilot Analytics',
            'Analytics',
            'manage_options',
            'path-pilot-analytics',
            [$this, 'render_analytics_page']
        );

        // Add submenu page for Settings
        add_submenu_page(
            'path-pilot',                            // Parent slug
            'Path Pilot Settings',                   // Page title
            'Settings',                              // Menu title
            'manage_options',                        // Capability
            'path-pilot-settings',                   // Menu slug
            [$this, 'render_settings_page']          // Our Pro callback
        );
    }

    /**
     * Override the settings page callback to use our Pro version's render method.
     */
    public function override_settings_page_callback() {
        global $admin_page_hooks, $_wp_submenu_pages;

        // Check if the parent menu is registered
        if (isset($admin_page_hooks['path-pilot'])) {
            $parent_slug = 'path-pilot';
            $settings_slug = 'path-pilot-settings';

            // Remove the parent's settings page callback
            if (isset($_wp_submenu_pages[$parent_slug])) {
                foreach ($_wp_submenu_pages[$parent_slug] as $key => $submenu_page) {
                    if ($submenu_page[2] === $settings_slug) {
                        unset($_wp_submenu_pages[$parent_slug][$key]);
                        break;
                    }
                }
            }

            // Add our own settings page callback with the same slug
            add_submenu_page(
                $parent_slug,                           // Parent slug
                'Path Pilot Settings',                  // Page title
                'Settings',                             // Menu title
                'manage_options',                       // Capability
                $settings_slug,                         // Menu slug
                [$this, 'render_settings_page']         // Our Pro callback
            );
        }
    }

    /**
     * Handle license key changes and schedule embeddings refresh
     */
    public function handle_license_key_change() {
        // Only run if this is our settings form
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'path_pilot_save_settings')) {
            return;
        }

        // Get the new API key from POST data
        $new_api_key = isset($_POST['path_pilot_api_key']) ? trim((string) wp_unslash($_POST['path_pilot_api_key'])) : '';
        $new_api_key = sanitize_text_field($new_api_key);
        $old_api_key = get_option('path_pilot_api_key', '');

        $api_key_changed = ($new_api_key !== $old_api_key);
        $redirect_query = [
            'page' => 'path-pilot-settings',
        ];

        // If a new key is provided, validate it first
        if (!empty($new_api_key)) {
            $api_status = Path_Pilot_Pro::get_api_key_status($new_api_key); // Pass new key for validation

            if (!$api_status['valid']) {
                // Key is invalid: do NOT save, show error, do NOT schedule embeddings
                Log::error('Path Pilot: Invalid API key submitted: ' . $api_status['message']);
                $redirect_query['updated'] = 'true';
                $redirect_query['notice_type'] = 'error';
                $redirect_query['message'] = 'API Key invalid: ' . $api_status['message'] . ' Please check your key.';
                
                $redirect_url = add_query_arg($redirect_query, admin_url('admin.php'));
                wp_safe_redirect($redirect_url);
                exit;
            }
        } else if (isset($_POST['path_pilot_api_key']) && $old_api_key !== '') {
            // If key was explicitly cleared by user
            $api_key_changed = true; // Mark as changed to trigger cleanup if needed
        }

        // If key is valid OR was explicitly cleared:
        if ($api_key_changed) {
            update_option('path_pilot_api_key', $new_api_key);

            if (!empty($new_api_key)) {
                // Only schedule embeddings if a valid key was saved
                if (!wp_next_scheduled('path_pilot_recompute_embeddings_full')) {
                    wp_schedule_single_event(time() + 5, 'path_pilot_recompute_embeddings_full');
                    set_transient('path_pilot_embeddings_notice', time(), 5 * MINUTE_IN_SECONDS);
                }
                $redirect_query['license'] = 'activated';
            }
        }

        // Redirect with success message (or default update message)
        $redirect_query['updated'] = 'true';
        $redirect_url = add_query_arg($redirect_query, admin_url('admin.php'));
        wp_safe_redirect($redirect_url);
        exit;
    }

    /**
     * Render the settings page with Pro license block at the top (server-side order)
     */
    public function render_settings_page() {
        // Ensure admin styles are available
        wp_enqueue_style('path-pilot-admin-style');

        echo '<div class="pp-content">';

        // NOTIFICATIONS SECTION - Always at the very top
        $this->render_notifications_section();

        // SINGLE FORM CONTAINING ALL SETTINGS
        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" id="path-pilot-settings-form">';
        echo '<input type="hidden" name="action" value="path_pilot_save_settings" />';
        wp_nonce_field('path_pilot_save_settings');

        // TOP CONTROLS SECTION - License key, API status, etc.
        $this->render_top_controls_section();

        // MAIN CONTENT SECTION - All the settings
        $this->render_main_content_section();

        // BOTTOM CONTROLS SECTION - Save button, etc.
        $this->render_bottom_controls_section();

        echo '</form>';

        // Loading overlay (copied from free)
        echo '<div id="pp-loading-overlay" style="display:none;position:fixed;top:0;left:0;width:100%;height:100%;background:rgba(0,0,0,0.5);z-index:9999;justify-content:center;align-items:center;">'
            .'<div style="background:#fff;padding:30px;border-radius:8px;text-align:center;box-shadow:0 4px 20px rgba(0,0,0,0.3);">'.'<div style="width:40px;height:40px;border:4px solid #f3f3f3;border-top:4px solid #1976d2;border-radius:50%;animation:spin 1s linear infinite;margin:0 auto 15px;"></div>'.'<div style="font-size:16px;font-weight:600;color:#333;">Saving Settings...</div>'.'</div>'.'</div>';

        echo '</div>';
    }

    /**
     * Render notifications section at the top
     */
    private function render_notifications_section() {
        echo '<div class="pp-notifications-section" style="margin-bottom: 20px;">';
        
        // Success notice with custom message support
        if (isset($_GET['updated']) && $_GET['updated'] === 'true') {
            $message = isset($_GET['message']) ? sanitize_text_field($_GET['message']) : 'Settings saved successfully!';
            $notice_type = isset($_GET['notice_type']) ? sanitize_text_field($_GET['notice_type']) : 'success';
            echo '<div class="notice notice-' . esc_attr($notice_type) . ' is-dismissible" style="margin: 0 0 12px 0;">';
            echo '<p>' . esc_html($message) . '</p>';
            echo '</div>';
        }

        // License activated notice
        if (isset($_GET['license']) && $_GET['license'] === 'activated') {
            echo '<div class="notice notice-success is-dismissible" style="margin: 0 0 12px 0;">';
            echo '<p>License key saved. Embeddings refresh has been scheduled.</p>';
            echo '</div>';
        }

        // Embeddings refresh notice (from transient)
        if (get_transient('path_pilot_embeddings_notice')) {
            echo '<div class="notice notice-info is-dismissible" style="margin: 0 0 12px 0;">';
            echo '<p><strong>Embeddings refresh started.</strong> This may take a few minutes in the background. You can continue using the admin while we rebuild.</p>';
            echo '</div>';
        }

        echo '</div>';
    }

    /**
     * Render top controls section (license key, API status, etc.)
     */
    private function render_top_controls_section() {
        echo '<div class="pp-top-controls-section" style="margin-bottom: 30px;">';
        
        // Pro license settings - instantiate the class and call the method
        $pro_settings = new Path_Pilot_Settings_Pro();
        $pro_settings->render_pro_settings();
        
        echo '</div>';
    }

    /**
     * Render main content section (all the settings)
     */
    private function render_main_content_section() {
        echo '<div class="pp-main-content-section">';
        
        // Shared settings content (no form wrapper needed - already inside main form)
        include_once dirname(dirname(dirname(__DIR__))) . '/path-pilot/admin/common/settings-common.php';
        
        echo '</div>';
    }

    /**
     * Render bottom controls section (save button, etc.)
     */
    private function render_bottom_controls_section() {
        echo '<div class="pp-bottom-controls-section" style="margin-top: 30px; padding-top: 20px; border-top: 1px solid #e5e5e5;">';
        
        echo '<button type="submit" class="btn btn-primary" id="save-settings-btn" style="padding:12px 24px;font-weight:600;font-size:1rem;">Save Settings</button>';
        
        echo '</div>';
    }

    public function pro_features_active() {
        // Check if Pro features are configured
        $api_status = Path_Pilot_Pro::get_api_key_status();
        return (bool) $api_status['valid'];
    }

    /**
     * Show admin notice for Pro setup on Path Pilot pages
     */
    public function show_pro_setup_notice() {
        // Only show on Path Pilot pages
        if (!$this->is_path_pilot_screen()) {
            return;
        }

        if (!Path_Pilot::is_pro()) {
            return;
        }

        // Don't show on settings page (they're already there)
        $screen = get_current_screen();
        if ($screen && $screen->id === 'path-pilot_page_path-pilot-settings') {
            return;
        }

        // Check if Pro features are configured
        $api_status = Path_Pilot_Pro::get_api_key_status();
        if ($this->pro_features_active()) {
            return; // Pro features are working, no notice needed
        }

        // Check if user has dismissed this notice
        $dismissed = get_user_meta(get_current_user_id(), 'path_pilot_setup_notice_dismissed', true);
        if ($dismissed) {
            return;
        }

        ?>
        <div class="notice notice-warning is-dismissible" id="path-pilot-setup-notice">
            <p>
                <strong>Path Pilot Pro Setup Required:</strong>
                <?php echo esc_html($api_status['message']); ?>
                <a href="<?php echo esc_url(admin_url('admin.php?page=path-pilot-settings')); ?>" class="button button-primary" style="margin-left: 10px;">
                    Configure License Key
                </a>
            </p>
        </div>

        <script>
            jQuery(document).ready(function($) {
                // Handle notice dismissal
                $(document).on('click', '#path-pilot-setup-notice .notice-dismiss', function() {
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'path_pilot_dismiss_setup_notice',
                            nonce: '<?php echo wp_create_nonce('path_pilot_dismiss_setup_notice'); ?>'
                        }
                    });
                });
            });
        </script>
        <?php
    }

    public function show_pro_status_message() {

        // Check Pro features status
        $api_status = Path_Pilot_Pro::get_api_key_status();
        if ($api_status['valid']) {
            return;
        }

        $pro_status_message = $api_status['message'];
        ?>
        <!-- Pro Features Status Notice -->
        <div class="pp-home-section pp-margin-bottom" style="border: 2px solid #f39c12; background: #fef9e7; border-radius: 8px; padding: 20px;">
            <h3 class="pp-section-heading" style="color: #e67e22; margin-bottom: 15px;"><i class="emoji-warning">⚠️</i> Pro Features Unavailable</h3>
            <div class="pp-home-protip" style="background: transparent; border: none; padding: 0; margin-bottom: 10px;">
                <i class="icon-pilot-icon" style="color: #e67e22;"></i>
                <strong>Status:</strong> Pro features are currently disabled. <?php echo esc_html($pro_status_message); ?>
            </div>
            <div style="margin-top: 15px;">
                <a href="<?php echo esc_url(admin_url('admin.php?page=path-pilot-settings')); ?>" class="btn btn-primary" style="padding: 10px 20px; font-weight: 600; text-decoration: none; display: inline-block;">
                    Configure License Key →
                </a>
            </div>
        </div>
        <?php
    }

    public function embedding_coverage($total_pages) {
        global $wpdb;
        $pages_with_embeddings = $wpdb->get_var("SELECT COUNT(DISTINCT post_id) FROM {$wpdb->prefix}path_pilot_vectors");
        $embedding_coverage = $total_pages > 0 ? round(($pages_with_embeddings / $total_pages) * 100) : 0;
        return [$embedding_coverage, $pages_with_embeddings];
    }

    /**
     * AJAX handler to get IDs of pages needing embeddings
     */
    public function ajax_get_pages_needing_embeddings() {
        check_ajax_referer('path_pilot_admin_ajax_nonce', '_wpnonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Not allowed']);
        }
        $ids = self::get_pages_needing_embeddings();
        wp_send_json_success(['ids' => $ids]);
    }

    /**
     * Get IDs of published pages that need embeddings (missing or expired)
     */
    public static function get_pages_needing_embeddings() {
        global $wpdb;

        // Check for full refresh flag
        $full_refresh = isset($_POST['full_refresh']) && $_POST['full_refresh'] === 'true';

        // Get allowed content types instead of all public post types
        $allowed_content_types = Path_Pilot_Shared::get_allowed_content_types();
        Log::info('Path Pilot Embeddings: Processing content types: ' . implode(', ', $allowed_content_types));

        // Get all published posts/pages of allowed types
        $all_posts = get_posts([
            'post_type' => $allowed_content_types,
            'post_status' => 'publish',
            'numberposts' => -1,
            'fields' => 'ids'
        ]);

        if ($full_refresh) {
            // Return all IDs for full refresh
            return $all_posts;
        }

        // For delta: Get existing embeddings
        $existing_embeddings = $wpdb->get_col("SELECT post_id FROM {$wpdb->prefix}path_pilot_vectors");

        // Find posts without embeddings
        $needing_embeddings = array_diff($all_posts, $existing_embeddings);

        return array_values($needing_embeddings);
    }

    /**
     * Background job to recompute embeddings for all allowed content types
     */
    public function recompute_all_embeddings_background() {
        $api_key = get_option('path_pilot_api_key', '');
        if (!$api_key) {
            Log::info('Path Pilot: Embeddings refresh skipped - no API key');
            return;
        }

        $allowed_content_types = Path_Pilot_Shared::get_allowed_content_types();
        Log::info('Path Pilot: Starting full embeddings refresh for types: ' . implode(', ', $allowed_content_types));

        $all_posts = get_posts([
            'post_type' => $allowed_content_types,
            'post_status' => 'publish',
            'numberposts' => -1,
            'fields' => 'ids'
        ]);

        $batch_size = 5;
        $processed = 0;
        $errors = 0;
        global $wpdb;
        foreach ($all_posts as $i => $id) {
            $item = get_post($id);
            if (!$item) continue;
            $text = $item->post_title . "\n\n" . $item->post_content;
            $embedding_result = Path_Pilot_Embeddings::get_openai_embedding($text, $api_key);
            if (is_array($embedding_result) && !empty($embedding_result['success']) && !empty($embedding_result['embedding'])) {
                $wpdb->replace($wpdb->prefix . 'path_pilot_vectors', [
                    'post_id' => $item->ID,
                    'embedding' => maybe_serialize($embedding_result['embedding']),
                    'updated_at' => current_time('mysql')
                ]);
            } else {
                $errors++;
            }
            $processed++;
            if ($processed % $batch_size === 0) {
                sleep(1); // basic pacing
            }
        }
        Log::info("Path Pilot: Full embeddings refresh complete. Processed={$processed}, Errors={$errors}");
    }

    /**
     * AJAX: schedule the embeddings refresh single event (unified trigger)
     */
    public function ajax_schedule_embeddings_refresh() {
        check_ajax_referer('path_pilot_admin_ajax_nonce', '_wpnonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Not allowed']);
        }
        if (!wp_next_scheduled('path_pilot_recompute_embeddings_full')) {
            wp_schedule_single_event(time() + 3, 'path_pilot_recompute_embeddings_full');
            set_transient('path_pilot_embeddings_notice', time(), 5 * MINUTE_IN_SECONDS);
        }
        wp_send_json_success(['scheduled' => true]);
    }

    /**
     * AJAX handler for batch embedding recompute
     */
    public function ajax_recompute_embeddings_batch() {
        check_ajax_referer('path_pilot_admin_ajax_nonce', '_wpnonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Not allowed']);
        }
        $api_key = get_option('path_pilot_api_key', '');
        if (!$api_key) {
            wp_send_json_error(['message' => 'No OpenAI key']);
        }
        $ids = isset($_POST['ids']) ? array_map('intval', (array)$_POST['ids']) : [];
        $batch_size = 5;
        $processed = 0;
        $errors = [];
        global $wpdb;
        foreach ($ids as $i => $id) {
            if ($processed >= $batch_size) break;
            $item = get_post($id);
            if (!$item) continue;
            $text = $item->post_title . "\n\n" . $item->post_content;
            $embedding_result = Path_Pilot_Embeddings::get_openai_embedding($text, $api_key);
            if (is_array($embedding_result) && !empty($embedding_result['success']) && !empty($embedding_result['embedding'])) {
                $wpdb->replace($wpdb->prefix . 'path_pilot_vectors', [
                    'post_id' => $item->ID,
                    'embedding' => maybe_serialize($embedding_result['embedding']),
                    'updated_at' => current_time('mysql')
                ]);
                Log::info("Path Pilot: Saved embedding for post ID $id");
            } else {
                $error_msg = is_array($embedding_result) && isset($embedding_result['error']) ? $embedding_result['error'] : 'Unknown error';
                Log::info("Path Pilot: Failed to get embedding for post ID $id. Error: $error_msg");
                $errors[$id] = $error_msg;
            }
            $processed++;
            sleep(1); // avoid rate limits
        }
        wp_send_json_success(['processed' => $processed, 'errors' => $errors]);
    }

    /**
     * Enqueue admin JS for embedding recompute
     */
    public function enqueue_admin_js($hook) {
        // Only load these scripts on our plugin's pages
        if (!$this->is_path_pilot_screen()) {
            return;
        }

        // For the settings page, load the main admin script with API access
        if ('path-pilot_page_path-pilot-settings' === $hook || 'toplevel_page_path-pilot' === $hook) {
            wp_enqueue_script(
                'path-pilot-admin-script',
                plugin_dir_url(__FILE__) . '../../admin/admin.js',
                ['jquery', 'wp-api'], // Add wp-api dependency for the nonce
                get_path_pilot_pro_version(),
                true
            );
        }

        // For the home/analytics page, load the embeddings script
        if ('toplevel_page_path-pilot' === $hook) {
            wp_enqueue_script(
                'path-pilot-embeddings-js',
                plugin_dir_url(__FILE__) . '../../admin/pro/embeddings.js',
                ['jquery'],
                get_path_pilot_pro_version(),
                true
            );
            wp_localize_script(
                'path-pilot-embeddings-js',
                'pp_admin_ajax',
                [
                    'ajax_url' => admin_url('admin-ajax.php'),
                    'nonce'    => wp_create_nonce('path_pilot_admin_ajax_nonce'),
                ]
            );
        }
    }

    public function render_embedding_coverage($total_pages) {
        $tuple = $this->embedding_coverage($total_pages);
        $embedding_coverage = $tuple[0];
        $pages_with_embeddings = $tuple[1];
        ?>
        <div class="pp-home-stat pp-stat-card">
            <div class="pp-home-stat-label">Embedding Coverage <span class="pp-pro-tag">PRO</span></div>
            <div id="pp-embedding-coverage-value" class="pp-home-stat-value"><?php echo $pages_with_embeddings; ?> / <?php echo $total_pages; ?></div>

            <?php if (!$this->pro_features_active()): ?>
                <!-- Pro Features Disabled -->
                <div class="pp-stat-waiting" style="color: #e67e22;">Pro features disabled</div>
                <div class="pp-progress-bar" style="opacity: 0.5;">
                    <div class="pp-progress-value" style="width: <?php echo $embedding_coverage; ?>%; background-color: #f39c12;"></div>
                </div>
                <div class="pp-stat-description" style="color: #e67e22;">
                    <strong>Semantic embeddings require a valid license key.</strong><br>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=path-pilot-settings')); ?>" style="color: #e67e22; text-decoration: underline;">Configure your license key</a> to enable AI-powered recommendations.
                </div>
            <?php else: ?>
                <!-- Single progress bar container that's always present -->
                <div id="pp-embedding-progress-container">
                    <div id="pp-embedding-status" class="pp-stat-waiting">
                        <?php if ($embedding_coverage < 100): ?>
                            Some pages are missing embeddings.
                        <?php else: ?>
                            100% coverage
                        <?php endif; ?>
                    </div>
                    <div class="pp-progress-bar">
                        <div id="pp-embedding-progress-bar" class="pp-progress-value" style="width: <?php echo $embedding_coverage; ?>%"></div>
                    </div>
                </div>

                <!-- Progress messages during sync -->
                <div id="pp-embedding-progress"></div>

                <!-- Status description for 100% coverage -->
                <?php if ($embedding_coverage >= 100): ?>
                    <div class="pp-stat-description">
                        All published pages have up-to-date semantic embeddings!
                    </div>
                <?php endif; ?>

                <!-- Buttons -->
                <div>
                    <?php if ($embedding_coverage < 100): ?>
                        <button id="pp-recompute-embeddings-ajax" class="btn btn-primary" style="padding:8px 18px;font-weight:600;font-size:1rem;margin-top:12px;">Sync</button>
                    <?php endif; ?>
                    <button id="pp-full-refresh-embeddings-ajax" class="btn btn-secondary" style="padding:8px 18px;font-weight:600;font-size:1rem;margin-top:12px;">Full Refresh</button>
                </div>
            <?php endif; ?>
        </div
        <?php
    }
}
