/**
 * Path Pilot Admin JavaScript
 */
jQuery(document).ready(function($) {
    console.log("Path Pilot Admin JS Loaded.");

    // Fetch and display credit balance on settings page
    if ($('#credit-balance-container').length) {
        console.log("Credit balance container found. Calling fetchCreditBalance().");
        fetchCreditBalance();
    }

    // Show success message when settings are saved
    if (window.location.search.includes('settings-updated=true')) {
        const notice = $('<div class="notice notice-success is-dismissible"><p>Settings saved successfully!</p></div>');
        $('.wrap h1').after(notice);

        // Auto-dismiss after 3 seconds
        setTimeout(function() {
            notice.fadeOut();
        }, 3000);
    }

    // Toggle password visibility for API key
    $('.path-pilot-admin').on('click', '.toggle-password', function(e) {
        e.preventDefault();
        const input = $($(this).data('toggle'));
        const icon = $(this).find('i');

        if (input.attr('type') === 'password') {
            input.attr('type', 'text');
            icon.removeClass('dashicons-visibility').addClass('dashicons-hidden');
            $(this).attr('title', 'Hide API Key');
        } else {
            input.attr('type', 'password');
            icon.removeClass('dashicons-hidden').addClass('dashicons-visibility');
            $(this).attr('title', 'Show API Key');
        }
    });

    // Add password toggle to API key input
    const apiKeyInput = $('#path_pilot_api_key');
    if (apiKeyInput.length) {
        const toggleBtn = $('<button type="button" class="toggle-password button-link" data-toggle="#path_pilot_api_key" title="Show API Key"><span class="screen-reader-text">Show API Key</span><i class="dashicons dashicons-visibility"></i></button>');
        apiKeyInput.after(toggleBtn);
        apiKeyInput.css('padding-right', '40px');
        toggleBtn.css({
            position: 'absolute',
            right: '10px',
            top: '50%',
            transform: 'translateY(-50%)',
            background: 'none',
            border: 'none',
            cursor: 'pointer'
        });
        apiKeyInput.parent().css('position', 'relative');
    }

    // Initialize tabs if present in URL
    const urlParams = new URLSearchParams(window.location.search);
    const tab = urlParams.get('tab');
    if (tab) {
        $('.nav-link').removeClass('active');
        $(`.nav-link[href*="tab=${tab}"]`).addClass('active');
    }

    // Add confirm dialog to settings reset button if it exists
    $('#reset-settings').on('click', function(e) {
        if (!confirm('Are you sure you want to reset all Path Pilot settings to defaults? This cannot be undone.')) {
            e.preventDefault();
        }
    });

    // Handle Remove Key button with confirmation
    $(document).on('click', '.pp-remove-key-btn', function(e) {
        e.preventDefault();
        const button = $(this);
        const confirmation = button.closest('.pp-api-key-display').find('.pp-remove-key-confirmation');

        if (!button.data('confirmed')) {
            // First click, show confirmation
            $('.pp-remove-key-confirmation').slideUp(); // Hide others
            $('.pp-remove-key-btn').data('confirmed', false); // Reset others

            confirmation.slideDown();
            button.data('confirmed', true);
        } else {
            // Second click, proceed with deletion
            confirmation.text('Removing key...').slideDown();

            const nonce = typeof wpApiSettings !== 'undefined' ? wpApiSettings.nonce : null;
            if (!nonce) {
                alert('Error: Auth Nonce not found. Cannot remove key.');
                button.data('confirmed', false);
                confirmation.slideUp();
                return;
            }

            $.ajax({
                url: '/wp-json/path-pilot/v1/delete-api-key',
                method: 'POST',
                contentType: 'application/json',
                beforeSend: function (xhr) {
                    xhr.setRequestHeader('X-WP-Nonce', nonce);
                },
                success: function(response) {
                    location.reload();
                },
                error: function(jqXHR) {
                    confirmation.text('Error deleting key. Please try again.').slideDown();
                    button.data('confirmed', false);
                    console.error('Error deleting API key:', jqXHR);
                }
            });
        }
    });

    // Handle Cancel Change button
    $(document).on('click', '.pp-cancel-change-btn', function(e) {
        e.preventDefault();

        const container = $(this).closest('.pp-home-stat-card');
        const displayDiv = container.find('.pp-api-key-display');
        const editDiv = container.find('.pp-api-key-edit');
        const input = editDiv.find('input[type="password"], input[type="text"]');

        // Clear the input and reset to password type
        input.val('').attr('type', 'password');

        // Reset toggle button
        const toggleBtn = editDiv.find('.pp-password-toggle');
        toggleBtn.removeClass('visible').attr('title', 'Show API Key');
        toggleBtn.find('.pp-eye-icon').text('👁️');

        // Show display, hide edit
        editDiv.hide();
        displayDiv.show();
    });

    // Hide confirmation messages if user clicks elsewhere
    $(document).on('click', function(e) {
        if (!$(e.target).closest('.pp-remove-key-btn').length && !$(e.target).closest('.pp-remove-key-confirmation').length) {
            $('.pp-remove-key-confirmation').slideUp();
            $('.pp-remove-key-btn').data('confirmed', false);
        }
    });

    // Store original API key value for comparison
    $('#path_pilot_api_key').each(function() {
        $(this).data('original-value', $(this).val());
    });

    /**
     * Shows or hides purchase buttons based on subscription status
     */
    function handlePurchaseButtonsVisibility(hasActiveSubscription, subscriptionId) {
        const purchaseButtonsContainer = $('.credit-purchase-options');
        const buyMoreHeading = purchaseButtonsContainer.prev('h5'); // The "Buy More Credits" heading
        const subscriptionPrompt = $('#subscription-prompt');
        const manageAccountContainer = $('#manage-account-container');

        if (hasActiveSubscription && subscriptionId) {
            // Show purchase buttons for active subscribers
            purchaseButtonsContainer.show();
            buyMoreHeading.show();
            manageAccountContainer.show();
            if (subscriptionPrompt.length) {
                subscriptionPrompt.hide();
            }
            console.log('Showing purchase buttons for subscription:', subscriptionId);
        } else {
            // Hide purchase buttons and show subscription prompt for non-subscribers
            purchaseButtonsContainer.hide();
            buyMoreHeading.hide();
            manageAccountContainer.hide();

            if (subscriptionPrompt.length === 0) {
                // Create subscription prompt if it doesn't exist
                const promptHtml = `
                    <div id="subscription-prompt" style="margin: 10px 0; padding: 12px; background: #f0f8ff; border: 1px solid #b3d9ff; border-radius: 4px;">
                        <p style="margin: 0; color: #2c3e50; font-size: 14px;">
                            <strong>💡 Need more credits?</strong>
                            <a href="https://pathpilot.app/pro/" target="_blank" style="color: #0073aa; text-decoration: none; font-weight: 600;">Subscribe to Path Pilot</a>
                            to purchase additional anytime credits that never expire.
                        </p>
                    </div>
                `;
                buyMoreHeading.after(promptHtml);
            } else {
                subscriptionPrompt.show();
            }
            console.log('Hiding purchase buttons - no active subscription');
        }
    }

    /**
     * Fetches the credit balance from the REST API and updates the display.
     */
    function fetchCreditBalance() {
        console.log("Inside fetchCreditBalance().");
        const loadingElem = $('#credit-balance-loading');
        const displayContainer = $('#credit-balance-display');
        const progressBarContainer = displayContainer.find('.progress-bar').first(); // Correctly target the outer container
        const progressBarFill = $('#credit-progress'); // The inner fill element
        const textElem = displayContainer.find('p').first();
        const breakdownContainer = $('#credit-breakdown');
        const breakdownTextElem = $('#credit-breakdown-text');

        // Use the global nonce provided by WordPress for authentication
        const nonce = typeof wpApiSettings !== 'undefined' ? wpApiSettings.nonce : null;
        console.log("WP API Nonce value is:", nonce);

        if (!nonce) {
            loadingElem.text('Error: Auth Nonce not found.');
            console.error('WP API Settings or Nonce not available. Make sure wp-api is an enqueued script dependency.');
            return;
        }

        console.log("Nonce found. Proceeding with AJAX request to /credits endpoint.");
        $.ajax({
            url: '/wp-json/path-pilot/v1/credits',
            method: 'GET',
            beforeSend: function (xhr) {
                console.log("Making AJAX call, setting X-WP-Nonce header.");
                xhr.setRequestHeader('X-WP-Nonce', nonce);
            },
            success: function(response) {
                console.log("AJAX success. Received data:", response);

                // Check if we have the new multi-key structure
                if (response && response.monthly && response.anytime) {
                    const monthlyRemaining = response.monthly.remaining;
                    const monthlyTotal = response.monthly.total;
                    const anytimeRemaining = response.anytime.remaining;
                    const anytimeTotal = response.anytime.total;
                    const totalRemaining = response.total_remaining;
                    const totalCredits = response.total_credits;
                    const hasActiveSubscription = response.has_active_subscription;
                    const subscriptionId = response.subscription_id;

                    const percentage = totalCredits > 0 ? (totalRemaining / totalCredits) * 100 : 0;

                    // Format numbers
                    const formattedMonthlyRemaining = monthlyRemaining.toLocaleString();
                    const formattedMonthlyTotal = monthlyTotal.toLocaleString();
                    const formattedAnytimeRemaining = anytimeRemaining.toLocaleString();
                    const formattedAnytimeTotal = anytimeTotal.toLocaleString();
                    const formattedTotalRemaining = totalRemaining.toLocaleString();
                    const formattedTotalCredits = totalCredits.toLocaleString();

                    // Hide loading and show display
                    loadingElem.hide();
                    displayContainer.show();

                    // Build credit text and breakdown
                    let creditText;
                    if (anytimeTotal > 0) {
                        // Show total when anytime credits exist
                        creditText = `Total: ${formattedTotalRemaining} / ${formattedTotalCredits} Credits Remaining`;

                        // Show breakdown with tooltip
                        breakdownTextElem.text(`Monthly: ${formattedMonthlyRemaining} / ${formattedMonthlyTotal} | Anytime: ${formattedAnytimeRemaining} / ${formattedAnytimeTotal}`);
                        breakdownContainer.show();

                        // Update tooltip for anytime credits
                        const helpTextElem = $('#credit-help-text');
                        helpTextElem.html(`
                            <strong>Monthly:</strong> Subscription credits, reset monthly<br>
                            <strong>Anytime:</strong> Purchased credits, never expire
                        `);
                    } else {
                        // Only show monthly credits when no anytime credits exist
                        creditText = `Current Credits: ${formattedMonthlyRemaining} / ${formattedMonthlyTotal} Credits Remaining`;
                        breakdownContainer.hide();
                    }

                    // Update text and progress bar
                    const creditAmountElem = $('#credit-amount');
                    if (creditAmountElem.length) {
                        creditAmountElem.text(formattedTotalRemaining);
                        console.log("#credit-amount updated with:", formattedTotalRemaining);
                    } else {
                        console.error("#credit-amount element not found for update.");
                    }

                    const creditParagraphElem = displayContainer.find('p').first();
                    if (creditParagraphElem.length) {
                        creditParagraphElem.html(`Current Credits: <strong id="credit-amount" style="font-size: 1.1em; color: #333;">${formattedTotalRemaining}</strong>`);
                        console.log("Credit paragraph updated with:", creditParagraphElem.html());
                    } else {
                        console.error("Credit display paragraph (<p>) not found for update.");
                    }

                    if (progressBarContainer.length) { // Check the container
                        progressBarContainer.show(); // Make the container visible
                        progressBarFill.css('width', percentage + '%'); // Update the fill
                        console.log("Progress bar container shown, fill updated with:", percentage + '%');
                    } else {
                        console.error("Progress bar container NOT found for update.");
                    }

                    // Change progress bar color based on usage
                    if (percentage < 10) {
                        progressBarFill.css('background-color', '#d63638'); // Red
                    } else if (percentage < 25) {
                        progressBarFill.css('background-color', '#f9a825'); // Yellow
                    } else {
                        progressBarFill.css('background-color', '#2ecc40'); // Green
                    }

                    // Show/hide purchase buttons based on subscription status
                    handlePurchaseButtonsVisibility(hasActiveSubscription, subscriptionId);

                } else if (response && typeof response.remaining !== 'undefined' && typeof response.total !== 'undefined') {
                    // Fallback for old single-key structure
                    const remaining = response.remaining;
                    const total = response.total;
                    const percentage = total > 0 ? (remaining / total) * 100 : 0;

                    const formattedRemaining = remaining.toLocaleString();
                    const formattedTotal = total.toLocaleString();

                    // Hide loading and show display
                    loadingElem.hide();
                    displayContainer.show();

                    // Update text and progress bar
                    const creditTextDisplayElem = displayContainer.find('p').first(); // Correctly target the <p> element
                    const progressBarElem = $('#credit-progress'); // The inner fill element
                    
                    console.log("Updating fallback elements: <p> length=", creditTextDisplayElem.length, "#credit-progress length=", progressBarElem.length);

                    if (creditTextDisplayElem.length) { creditTextDisplayElem.html(`Current Credits: <strong id="credit-amount" style="font-size: 1.1em; color: #333;">${formattedRemaining}</strong>`); }
                    if (progressBarContainer.length) { // Check the container
                        progressBarContainer.show(); // Make the container visible
                        progressBarFill.css('width', percentage + '%'); // Update the fill
                        console.log("Fallback Progress bar container shown, fill updated with:", percentage + '%');
                    } else {
                        console.error("Fallback Progress bar container NOT found for update.");
                    }
                    breakdownContainer.hide();

                    // Change progress bar color based on usage
                    if (percentage < 10) {
                        progressBarFill.css('background-color', '#d63638'); // Red
                    } else if (percentage < 25) {
                        progressBarFill.css('background-color', '#f9a825'); // Yellow
                    } else {
                        progressBarFill.css('background-color', '#2ecc40'); // Green
                    }

                    // For old structure, assume no active subscription
                    handlePurchaseButtonsVisibility(false, '');
                    console.log("handlePurchaseButtonsVisibility called for fallback.");

                } else {
                    console.log("No valid credit data structure received.");
                    loadingElem.hide();
                    displayContainer.show();
                    textElem.html('N/A');
                    progressBarContainer.hide(); // Hide the container if no data
                    console.log("No valid data received, hiding progress bar container.");

                    // Hide purchase buttons when no valid data
                    handlePurchaseButtonsVisibility(false, '');
                    console.log("handlePurchaseButtonsVisibility called for no data.");
                }
            },
            error: function(jqXHR) {
                let errorMessage = 'Error loading credits';
                if (jqXHR.responseJSON && jqXHR.responseJSON.error) {
                    errorMessage = jqXHR.responseJSON.error;
                } else if (jqXHR.status === 403) {
                    errorMessage = 'Permission Denied';
                } else if (jqXHR.status === 500) {
                    errorMessage = 'Server error - please try again later';
                }

                console.error("AJAX Error:", errorMessage, jqXHR);

                loadingElem.hide();
                displayContainer.show();
                textElem.html(errorMessage).css({
                    'color': '#d63638'
                });
                progressBarContainer.hide(); // Hide the container on error
                console.log("AJAX error, hiding progress bar container.");

                // Hide purchase buttons on error (no subscription status available)
                handlePurchaseButtonsVisibility(false, '');
            }
        });
    }

    // Manage Account button click -> request portal session and redirect
    $(document).on('click', '#pp-manage-account', function(e) {
        e.preventDefault();
        const nonce = typeof wpApiSettings !== 'undefined' ? wpApiSettings.nonce : null;
        if (!nonce) {
            alert('Authentication error. Please refresh the page and try again.');
            return;
        }
        const button = $(this);
        const originalText = button.text();
        button.prop('disabled', true).text('Opening portal...');
        $.ajax({
            url: '/wp-json/path-pilot/v1/create-portal-session',
            method: 'POST',
            contentType: 'application/json',
            data: JSON.stringify({}),
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-WP-Nonce', nonce);
            },
            success: function(response) {
                if (response && response.url) {
                    window.location.href = response.url;
                } else {
                    alert('Could not open the billing portal. Please try again later.');
                    button.prop('disabled', false).text(originalText);
                }
            },
            error: function(jqXHR) {
                // Log full error details to help debugging without server logs
                try {
                    console.error('Portal session error:', {
                        status: jqXHR.status,
                        statusText: jqXHR.statusText,
                        responseText: jqXHR.responseText,
                        responseJSON: jqXHR.responseJSON
                    });
                } catch (e) {}
                const hasJSON = jqXHR.responseJSON && typeof jqXHR.responseJSON === 'object';
                const msg = hasJSON && jqXHR.responseJSON.error ? jqXHR.responseJSON.error : 'Failed to create portal session.';
                const fallback = hasJSON && jqXHR.responseJSON.fallback_url ? jqXHR.responseJSON.fallback_url : null;
                if (fallback) {
                    if (confirm(msg + '\n\nOpen account portal now?')) {
                        window.open(fallback, '_blank');
                    }
                } else {
                    alert(msg);
                }
                button.prop('disabled', false).text(originalText);
            }
        });
    });

    /**
     * Tests an API key by attempting to fetch credit balance
     */
    function testApiKey(apiKey, resultContainer) {
        if (!apiKey || apiKey.trim() === '') {
            showTestResult(resultContainer, false, 'Please enter an API key first.');
            return;
        }

        // Show loading state
        showTestResult(resultContainer, null, 'Testing API key connection...');

        const nonce = typeof wpApiSettings !== 'undefined' ? wpApiSettings.nonce : null;
        console.log("WP API Nonce value in testApiKey:", nonce);
        if (!nonce) {
            showTestResult(resultContainer, false, 'Authentication error: No nonce available.');
            return;
        }

        $.ajax({
            url: '/wp-json/path-pilot/v1/test-api-key',
            method: 'POST',
            contentType: 'application/json',
            data: JSON.stringify({
                api_key: apiKey
            }),
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-WP-Nonce', nonce);
            },
            success: function(response) {
                if (response.valid) {
                    const credits = response.credits;
                    const subscription = response.subscription;

                    let message = 'API key is valid and working! ';
                    if (credits && credits.total > 0) {
                        message += `You have ${credits.remaining.toLocaleString()} of ${credits.total.toLocaleString()} credits remaining.`;
                    }
                    if (subscription && subscription.has_active_subscription) {
                        message += ' Active subscription detected.';
                    }

                    showTestResult(resultContainer, true, message);
                } else {
                    showTestResult(resultContainer, false, response.error || 'API key validation failed.');
                }
            },
            error: function(jqXHR) {
                let errorMessage = 'API key test failed';
                if (jqXHR.responseJSON && jqXHR.responseJSON.error) {
                    errorMessage = jqXHR.responseJSON.error;
                } else if (jqXHR.status === 400) {
                    errorMessage = 'Invalid request - please check your API key format';
                } else if (jqXHR.status === 403) {
                    errorMessage = 'Permission denied - please refresh the page and try again';
                } else if (jqXHR.status === 500) {
                    errorMessage = 'Server error - please try again later';
                }

                showTestResult(resultContainer, false, errorMessage);
            }
        });
    }

    /**
     * Shows the result of API key testing
     */
    function showTestResult(container, success, message) {
        container.show();

        if (success === null) {
            // Loading state
            container.removeClass('success error').addClass('loading');
            container.css({
                'background': '#f0f8ff',
                'border': '1px solid #b3d9ff',
                'color': '#2c3e50'
            });
            container.html('<strong>🔄 Testing...</strong> ' + message);
        } else if (success) {
            // Success state
            container.removeClass('loading error').addClass('success');
            container.css({
                'background': '#eafaf1',
                'border': '1px solid #2ecc40',
                'color': '#27ae60'
            });
            container.html('<strong>✅ Success!</strong> ' + message);
        } else {
            // Error state
            container.removeClass('loading success').addClass('error');
            container.css({
                'background': '#ffeaea',
                'border': '1px solid #ff6b6b',
                'color': '#d63031'
            });
            container.html('<strong>❌ Error:</strong> ' + message);
        }
    }

    // Handle API key testing button
    $('#test-api-key').on('click', function(e) {
        e.preventDefault();
        const apiKey = $('#path_pilot_api_key').val().trim();
        const resultContainer = $('#api-test-result');

        testApiKey(apiKey, resultContainer);
    });

    // Auto-test API key when field loses focus (if it has content)
    $('#path_pilot_api_key').on('blur', function() {
        const apiKey = $(this).val().trim();
        const resultContainer = $('#api-test-result');

        // Only auto-test if there's content and we're in the pre-subscription state
        if (apiKey && $('#path-pilot-api-key-form').length > 0) {
            // Small delay to avoid testing while user is still typing
            setTimeout(function() {
                testApiKey(apiKey, resultContainer);
            }, 500);
        }
    });

    // Store original API key value for comparison
    $('#path_pilot_api_key').each(function() {
        $(this).data('original-value', $(this).val());
    });
})(jQuery);
