// Path Pilot Pro - Chat functionality
(function() {
    console.log("this is chat.js");
  // Wait for DOM to be ready
  document.addEventListener('DOMContentLoaded', function() {
    initPathPilotChat();
  });

  function initPathPilotChat() {
    // Only initialize if Pro is enabled and chat is enabled
    if (!window.PathPilotStatus || !window.PathPilotStatus.is_pro || !window.PathPilotStatus.chat_enabled) {
      console.log('Path Pilot Pro: Chat not enabled - missing requirements');
      return;
    }

    console.log('Path Pilot Pro: Creating chat interface...');
    createChatInterface();
  }

  function createChatInterface() {
    // Find the existing drawer content
    const drawerContent = document.querySelector('.pp-drawer-content');
    if (!drawerContent) {
      console.error('Path Pilot Pro: Could not find drawer content');
      return;
    }

    // --- Chat History Expiry (24hr) ---
    const CHAT_HISTORY_KEY = 'path_pilot_chat_history';
    const CHAT_HISTORY_EXPIRY_KEY = 'path_pilot_chat_history_expiry';

    function saveChatHistory() {
      try {
        localStorage.setItem(CHAT_HISTORY_KEY, JSON.stringify(chatHistory));
        localStorage.setItem(CHAT_HISTORY_EXPIRY_KEY, Date.now().toString());
      } catch (e) {}
    }

    function loadChatHistory() {
      try {
        const expiry = parseInt(localStorage.getItem(CHAT_HISTORY_EXPIRY_KEY), 10);
        if (expiry && Date.now() - expiry > 24 * 60 * 60 * 1000) {
          localStorage.removeItem(CHAT_HISTORY_KEY);
          localStorage.removeItem(CHAT_HISTORY_EXPIRY_KEY);
          return [];
        }
        const saved = localStorage.getItem(CHAT_HISTORY_KEY);
        if (saved) return JSON.parse(saved);
      } catch (e) {}
      return [];
    }

    let chatHistory = loadChatHistory();

    // Create separator
    const recChatSeparator = document.createElement('div');
    recChatSeparator.className = 'pp-rec-chat-separator';
    drawerContent.appendChild(recChatSeparator);

    // Create chat section
    const chatSection = document.createElement('div');
    chatSection.className = 'pp-drawer-chat';
    chatSection.innerHTML = `
      <div class="pp-chat-history-container">
        <div class="pp-chat-history"></div>
      </div>
      <div class="pp-chat-tools"><a href="#" class="pp-chat-clear">Clear history</a></div>
      <form class="pp-chat-input-container" autocomplete="off" onsubmit="return false;">
        <textarea class="pp-drawer-chat-input" placeholder="What are you looking for?" rows="1"></textarea>
        <button class="pp-drawer-chat-send" aria-label="Send" type="submit" disabled>
          <svg width="22" height="22" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><line x1="22" y1="2" x2="11" y2="13"></line><polygon points="22 2 15 22 11 13 2 9 22 2"></polygon></svg>
        </button>
      </form>
    `;
    drawerContent.appendChild(chatSection);

    // Get chat elements
    const chatForm = chatSection.querySelector('.pp-chat-input-container');
    const chatInput = chatForm.querySelector('.pp-drawer-chat-input');
    const chatSend = chatForm.querySelector('.pp-drawer-chat-send');
    const chatHistoryDiv = chatSection.querySelector('.pp-chat-history');
    const chatHistoryContainer = chatSection.querySelector('.pp-chat-history-container');
    const chatClear = chatSection.querySelector('.pp-chat-clear');

    // Setup chat functionality
    setupChatFunctionality(chatForm, chatInput, chatSend, chatHistoryDiv, chatHistoryContainer, chatClear, chatHistory, saveChatHistory);

    function setupChatFunctionality(chatForm, chatInput, chatSend, chatHistoryDiv, chatHistoryContainer, chatClear, chatHistory, saveChatHistory) {
      function renderChatHistory() {
        chatHistoryDiv.innerHTML = '';
        chatHistory.forEach(msg => {
          const row = document.createElement('div');
          row.className = 'pp-chat-row pp-chat-' + msg.role;
          if (msg.role === 'assistant') {
            row.innerHTML = `<div class="pp-chat-bubble pp-chat-assistant">${msg.content}</div>`;
          } else {
            row.innerHTML = `<div class="pp-chat-bubble pp-chat-user">${msg.content}</div>`;
          }
          chatHistoryDiv.appendChild(row);
        });
        // If waiting for response, show thinking animation
        if (window.ppChatThinking) {
          const thinkingRow = document.createElement('div');
          thinkingRow.className = 'pp-chat-thinking';
          thinkingRow.innerHTML = `
            <div class="pp-chat-bubble">
              <span class="pp-thinking-text">Thinking</span>
              <div class="pp-thinking-dots">
                <div class="pp-thinking-dot"></div>
                <div class="pp-thinking-dot"></div>
                <div class="pp-thinking-dot"></div>
              </div>
            </div>
          `;
          chatHistoryDiv.appendChild(thinkingRow);
        }
        // Always scroll to bottom
        setTimeout(() => { chatHistoryContainer.scrollTop = chatHistoryContainer.scrollHeight; }, 50);
      }

      renderChatHistory(); // Initial render

      // Chat clear functionality
      chatClear.addEventListener('click', function(e) {
        e.preventDefault();
        chatHistory.length = 0; // Clear array
        saveChatHistory();
        renderChatHistory();
      });

      // Event Listeners for chat
      chatInput.addEventListener('keydown', function(e) {
        if (e.key === 'Enter' && !e.shiftKey && !e.ctrlKey) {
          e.preventDefault();
          if (!chatSend.disabled) chatSend.click();
        }
      });

      chatInput.addEventListener('input', function() {
        chatSend.disabled = !chatInput.value.trim();
        chatInput.style.height = 'auto';
        chatInput.style.height = (chatInput.scrollHeight) + 'px';
      });

      chatForm.addEventListener('submit', function(e) {
        e.preventDefault();
        const userMessage = chatInput.value.trim();
        if (!userMessage) return;

        chatHistory.push({ role: 'user', content: userMessage });
        chatInput.value = '';
        chatInput.style.height = 'auto';
        chatSend.disabled = true;
        window.ppChatThinking = true;
        renderChatHistory();

        // Send the current question separately from the history
        const previousHistory = chatHistory.slice(0, -1); // All messages except the current one

        fetch('/wp-json/path-pilot/v1/chat', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          credentials: 'same-origin',
          body: JSON.stringify({
            question: userMessage,
            has_seen_recommendation: true, // Chat is only available after seeing recommendations
            history: previousHistory
          })
        })
        .then(r => r.json())
        .then(data => {
          window.ppChatThinking = false;
          if (data && data.reply) {
            chatHistory.push({ role: 'assistant', content: data.reply });
          } else {
            chatHistory.push({ role: 'assistant', content: 'Sorry, I encountered an error.' });
          }
          renderChatHistory();
          saveChatHistory();
        })
        .catch(err => {
          window.ppChatThinking = false;
          chatHistory.push({ role: 'assistant', content: 'Sorry, I couldn\'t connect.' });
          renderChatHistory();
          saveChatHistory();
        });
      });
    }
  }

})();
