<?php
namespace Path_Pilot;

if (!defined('ABSPATH')) exit;

class Path_Pilot_Update_Check {

    private $plugin_slug;
    private $plugin_path;

    public function __construct($plugin_path) {
        $this->plugin_slug = plugin_basename($plugin_path);
        $this->plugin_path = $plugin_path;

        add_filter('pre_set_site_transient_update_plugins', [$this, 'check_for_update']);
        add_filter('plugins_api', [$this, 'plugin_info'], 20, 3);
    }

    public function check_for_update($transient) {
        if (empty($transient->checked)) {
            return $transient;
        }

        $remote_info = $this->get_remote_info();

        if ($remote_info && version_compare(get_path_pilot_pro_version(), $remote_info->version, '<')) {
            $transient->response[$this->plugin_slug] = (object)[
                'slug'        => $this->plugin_slug,
                'new_version' => $remote_info->version,
                'url'         => 'https://pathpilot.app',
                'package'     => 'https://pathpilot.app/downloads/path-pilot-pro.latest-stable.zip',
                'tested'      => '6.6',
                'sections'    => [
                    'changelog' => (new \Parsedown())->text($remote_info->changelog),
                ],
            ];
        }

        return $transient;
    }

    public function plugin_info($res, $action, $args) {
        // do nothing if this is not about getting plugin information
        if ($action !== 'plugin_information') {
            return false;
        }

        // do nothing if it is not our plugin
        if ($this->plugin_slug !== $args->slug) {
            return false;
        }

        $remote_info = $this->get_remote_info();

        if (!$remote_info) {
            return false;
        }

        $res = new \stdClass();
        $res->name = 'Path Pilot Pro';
        $res->slug = $this->plugin_slug;
        $res->version = $remote_info->version;
        $res->author = 'Path Pilot';
        $res->author_profile = 'https://pathpilot.app';
        $res->last_updated = $remote_info->release_date;
        $res->sections = [
            'description' => $remote_info->description,
            'changelog'   => (new \Parsedown())->text($remote_info->changelog),
        ];
        $res->download_link = 'https://pathpilot.app/download/path-pilot-pro.latest-stable.zip';

        return $res;
    }

    public function get_remote_info() {
        $response = wp_remote_get('https://pathpilot.app/wp-json/path-pilot/v1/downloads/dist/pro/current/');

        if (is_wp_error($response) || 200 !== wp_remote_retrieve_response_code($response)) {
            return false;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body);

        if (empty($data) || !isset($data->version)) {
            return false;
        }

        return $data;
    }
}
