<?php
namespace Path_Pilot;

// AI API integration for Path Pilot (Pro only)
if (!defined('ABSPATH')) exit;

class Path_Pilot_AI {
    public function __construct() {
        add_filter('path_pilot_rewrite_recommendation_descriptions', [ $this, 'rewrite_recommendation_descriptions' ]);
    }

    // Get an embedding from OpenAI
    public static function get_openai_embedding($text, $api_key) {
        $body = json_encode([
            'input' => $text,
            'model' => 'text-embedding-3-small'
        ]);
        $response = wp_remote_post(PATH_PILOT_AI_API_URL . '/embeddings', [
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $api_key
            ],
            'body' => $body,
            'timeout' => 20
        ]);
        if (is_wp_error($response)) {
            Log::info('Path Pilot: OpenAI request WP_Error: ' . $response->get_error_message());
            return false;
        }
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        if (isset($data['data'][0]['embedding'])) {
            return $data['data'][0]['embedding'];
        }
        // Log the full response if embedding is missing
        Log::info('Path Pilot: LiteLLM or Proxy Error: ' . $body);
        return false;
    }

    // Rewrite a query using OpenAI (for semantic search)
    public static function rewrite_query_with_openai($question, $api_key) {
        $rewrite_prompt = "Rewrite the following question as a concise search query for finding the most relevant help page on our website. Only output the search query, nothing else.\n\nQuestion: \"$question\"";
        $rewrite_body = json_encode([
            'model' => 'gem_flash',
            'messages' => [
                [ 'role' => 'system', 'content' => 'You are a helpful assistant.' ],
                [ 'role' => 'user', 'content' => $rewrite_prompt ]
            ],
            'max_tokens' => 32
        ]);
        $rewrite_response = wp_remote_post(PATH_PILOT_AI_API_URL . '/chat/completions', [
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $api_key
            ],
            'body' => $rewrite_body,
            'timeout' => 20
        ]);
        if (is_wp_error($rewrite_response)) return false;
        $rewrite_data = json_decode(wp_remote_retrieve_body($rewrite_response), true);
        $rewrite_reply = trim($rewrite_data['choices'][0]['message']['content'] ?? '');
        return $rewrite_reply ?: false;
    }

    // Chat with OpenAI (for chat endpoint)
    public static function chat_with_openai($prompt_or_params, $api_key, $max_tokens = 5000) {
        if (is_array($prompt_or_params) && isset($prompt_or_params['messages'])) {
            // Use the provided array directly
            $params = $prompt_or_params;
            if (!isset($params['model'])) {
                $params['model'] = 'gem_flash'; // or your default model
            }
            if (!isset($params['max_tokens'])) {
                $params['max_tokens'] = $max_tokens;
            }
        } else {
            // Use the old prompt string logic
            $params = [
                'model' => 'gem_flash',
                'messages' => [
                    [ 'role' => 'system', 'content' => 'You are a helpful website assistant providing recommendations for the user to visit on our site.' ],
                    [ 'role' => 'user', 'content' => $prompt_or_params ]
                ],
                'max_tokens' => $max_tokens
            ];
        }
        $body = json_encode($params);
        $response = wp_remote_post(PATH_PILOT_AI_API_URL . '/chat/completions', [
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $api_key
            ],
            'body' => $body,
            'timeout' => 30
        ]);
        if (is_wp_error($response)) {
            Log::info('Path Pilot AI: WP_Error in chat_with_openai: ' . $response->get_error_message());
            return false;
        }
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        if (!isset($data['choices'][0]['message']['content'])) {
            Log::info('Path Pilot AI: Unexpected response in chat_with_openai: ' . $body);
            return false;
        }
        return $data['choices'][0]['message']['content'];
    }

    // AI-powered promotional descriptions for recommendations
    public static function rewrite_recommendation_descriptions($context) {
        $api_key = get_option('path_pilot_api_key', '');
        if (empty($context) || empty($api_key)) return $context;

        $to_generate = [];
        $cached_promos = [];
        foreach ($context as $c) {
            $page_id = isset($c['page_id']) ? $c['page_id'] : (isset($c['id']) ? $c['id'] : '');
            if (!$page_id) continue;
            $cache = get_site_transient('pp_ai_promo_' . $page_id);
            if ($cache && isset($cache['title']) && isset($cache['synopsis'])) {
                $cached_promos[$page_id] = $cache;
            } else {
                $to_generate[] = $c;
            }
        }

        $promos = [];
        if (!empty($to_generate)) {
            $prompt = "You are a copywriter. For each of the following web pages, write a short, enticing promotional description (1-2 sentences) that encourages a user to visit the page. Return your response as a JSON array of objects, each with 'page_id' (pass through the page_id), 'title', and 'synopsis' fields.\n\nPages:";
            $i = 1;
            foreach ($to_generate as $c) {
                $page_id = isset($c['page_id']) ? $c['page_id'] : (isset($c['id']) ? $c['id'] : '');
                $title = isset($c['title']) ? $c['title'] : '';
                $synopsis = isset($c['synopsis']) ? $c['synopsis'] : '';
                $prompt .= "\n$i. page_id: {$page_id}\n   title: {$title}\n   synopsis: {$synopsis}";
                $i++;
            }
            $body = [
                'model' => 'gem_flash',
                'messages' => [
                    [ 'role' => 'system', 'content' => 'You are a helpful website copywriter.' ],
                    [ 'role' => 'user', 'content' => $prompt ]
                ],
                'max_tokens' => 1024
            ];
            $response = wp_remote_post(PATH_PILOT_AI_API_URL . '/chat/completions', [
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Authorization' => 'Bearer ' . $api_key
                ],
                'body' => json_encode($body),
                'timeout' => 30
            ]);
            if (!is_wp_error($response)) {
                $body = wp_remote_retrieve_body($response);
                $data = json_decode($body, true);
                $content = $data['choices'][0]['message']['content'] ?? '';
                if (preg_match('/\[.*\]/s', $content, $matches)) {
                    $json = $matches[0];
                    $promos = json_decode($json, true);
                } else {
                    $promos = json_decode($content, true);
                }
                Log::info('Recommendation Content: ' . $content);
                if (is_array($promos)) {
                    foreach ($promos as $promo) {
                        if (isset($promo['page_id'])) {
                            set_site_transient('pp_ai_promo_' . $promo['page_id'], $promo, 10 * DAY_IN_SECONDS);
                            $cached_promos[$promo['page_id']] = $promo;
                        }
                    }
                }
            }
        }

        // Merge cached promos into the context
        foreach ($context as $i => $rec) {
            $pid = isset($rec['page_id']) ? $rec['page_id'] : (isset($rec['id']) ? $rec['id'] : null);
            if ($pid && isset($cached_promos[$pid])) {
                if (!empty($cached_promos[$pid]['title'])) {
                    $context[$i]['title'] = $cached_promos[$pid]['title'];
                }
                if (!empty($cached_promos[$pid]['synopsis'])) {
                    $context[$i]['synopsis'] = $cached_promos[$pid]['synopsis'];
                }
            }
        }
        return $context;
    }
}
// In the free build, do not include this file or override with a stub that returns false/null for all methods.
