<?php
namespace Path_Pilot;

// Path Pilot Admin functionality
if (!defined('ABSPATH')) exit;

class Path_Pilot_Admin_Pro extends Path_Pilot_Admin {
    public function __construct() {

        // Add admin notices for Path Pilot pages
        add_action('admin_notices', [$this, 'show_pro_setup_notice']);
        add_action('path_pilot_show_pro_status_message', [$this, 'show_pro_status_message']);
        add_action('path_pilot_render_embedding_coverage', [$this, 'render_embedding_coverage']);

        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_js']);
        add_action('wp_ajax_get_pages_needing_embeddings', [$this, 'ajax_get_pages_needing_embeddings']);
        add_action('wp_ajax_recompute_embeddings_batch', [$this, 'ajax_recompute_embeddings_batch']);
    }

    public function pro_features_active() {
        // Check if Pro features are configured
        $api_status = Path_Pilot_Pro::get_api_key_status();
        return (bool) $api_status['valid'];
    }
    /**
     * Show admin notice for Pro setup on Path Pilot pages
     */
    public function show_pro_setup_notice() {
        // Only show on Path Pilot pages
        if (!$this->is_path_pilot_screen()) {
            return;
        }

        if (!Path_Pilot::is_pro()) {
            return;
        }

        // Don't show on settings page (they're already there)
        $screen = get_current_screen();
        if ($screen && $screen->id === 'path-pilot_page_path-pilot-settings') {
            return;
        }

        // Check if Pro features are configured
        $api_status = Path_Pilot_Pro::get_api_key_status();
        if ($this->pro_features_active()) {
            return; // Pro features are working, no notice needed
        }

        // Check if user has dismissed this notice
        $dismissed = get_user_meta(get_current_user_id(), 'path_pilot_setup_notice_dismissed', true);
        if ($dismissed) {
            return;
        }

        ?>
        <div class="notice notice-warning is-dismissible" id="path-pilot-setup-notice">
            <p>
                <strong>Path Pilot Pro Setup Required:</strong>
                <?php echo esc_html($api_status['message']); ?>
                <a href="<?php echo esc_url(admin_url('admin.php?page=path-pilot-settings')); ?>" class="button button-primary" style="margin-left: 10px;">
                    Configure License Key
                </a>
            </p>
        </div>

        <script>
            jQuery(document).ready(function($) {
                // Handle notice dismissal
                $(document).on('click', '#path-pilot-setup-notice .notice-dismiss', function() {
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'path_pilot_dismiss_setup_notice',
                            nonce: '<?php echo wp_create_nonce('path_pilot_dismiss_setup_notice'); ?>'
                        }
                    });
                });
            });
        </script>
        <?php
    }

    public function show_pro_status_message() {

        // Check Pro features status
        $api_status = Path_Pilot_Pro::get_api_key_status();
        if ($api_status['valid']) {
            return;
        }

        $pro_status_message = $api_status['message'];
        ?>
        <!-- Pro Features Status Notice -->
        <div class="pp-home-section pp-margin-bottom" style="border: 2px solid #f39c12; background: #fef9e7; border-radius: 8px; padding: 20px;">
            <h3 class="pp-section-heading" style="color: #e67e22; margin-bottom: 15px;"><i class="emoji-warning">⚠️</i> Pro Features Unavailable</h3>
            <div class="pp-home-protip" style="background: transparent; border: none; padding: 0; margin-bottom: 10px;">
                <i class="icon-pilot-icon" style="color: #e67e22;"></i>
                <strong>Status:</strong> Pro features are currently disabled. <?php echo esc_html($pro_status_message); ?>
            </div>
            <div style="margin-top: 15px;">
                <a href="<?php echo esc_url(admin_url('admin.php?page=path-pilot-settings')); ?>" class="btn btn-primary" style="padding: 10px 20px; font-weight: 600; text-decoration: none; display: inline-block;">
                    Configure License Key →
                </a>
            </div>
        </div>
        <?php
    }

    public function embedding_coverage($total_pages) {
        global $wpdb;
        $pages_with_embeddings = $wpdb->get_var("SELECT COUNT(DISTINCT post_id) FROM {$wpdb->prefix}path_pilot_vectors");
        $embedding_coverage = $total_pages > 0 ? round(($pages_with_embeddings / $total_pages) * 100) : 0;
        return [$embedding_coverage, $pages_with_embeddings];
    }

    /**
     * AJAX handler to get IDs of pages needing embeddings
     */
    public function ajax_get_pages_needing_embeddings() {
        check_ajax_referer('path_pilot_admin_ajax_nonce', '_wpnonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Not allowed']);
        }
        $ids = self::get_pages_needing_embeddings();
        wp_send_json_success(['ids' => $ids]);
    }

    /**
     * Get IDs of published pages that need embeddings (missing or expired)
     */
    public static function get_pages_needing_embeddings() {
        global $wpdb;

        // Check for full refresh flag
        $full_refresh = isset($_POST['full_refresh']) && $_POST['full_refresh'] === 'true';

        // Get allowed content types instead of all public post types
        $allowed_content_types = Path_Pilot_Shared::get_allowed_content_types();
        Log::info('Path Pilot Embeddings: Processing content types: ' . implode(', ', $allowed_content_types));

        // Get all published posts/pages of allowed types
        $all_posts = get_posts([
            'post_type' => $allowed_content_types,
            'post_status' => 'publish',
            'numberposts' => -1,
            'fields' => 'ids'
        ]);

        if ($full_refresh) {
            // Return all IDs for full refresh
            return $all_posts;
        }

        // For delta: Get existing embeddings
        $existing_embeddings = $wpdb->get_col("SELECT post_id FROM {$wpdb->prefix}path_pilot_vectors");

        // Find posts without embeddings
        $needing_embeddings = array_diff($all_posts, $existing_embeddings);

        return array_values($needing_embeddings);
    }

    /**
     * AJAX handler for batch embedding recompute
     */
    public function ajax_recompute_embeddings_batch() {
        check_ajax_referer('path_pilot_admin_ajax_nonce', '_wpnonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Not allowed']);
        }
        $api_key = get_option('path_pilot_api_key', '');
        if (!$api_key) {
            wp_send_json_error(['message' => 'No OpenAI key']);
        }
        $ids = isset($_POST['ids']) ? array_map('intval', (array)$_POST['ids']) : [];
        $batch_size = 5;
        $processed = 0;
        $errors = [];
        global $wpdb;
        foreach ($ids as $i => $id) {
            if ($processed >= $batch_size) break;
            $item = get_post($id);
            if (!$item) continue;
            $text = $item->post_title . "\n\n" . $item->post_content;
            $embedding_result = Path_Pilot_Embeddings::get_openai_embedding($text, $api_key);
            if (is_array($embedding_result) && !empty($embedding_result['success']) && !empty($embedding_result['embedding'])) {
                $wpdb->replace($wpdb->prefix . 'path_pilot_vectors', [
                    'post_id' => $item->ID,
                    'embedding' => maybe_serialize($embedding_result['embedding']),
                    'updated_at' => current_time('mysql')
                ]);
                Log::info("Path Pilot: Saved embedding for post ID $id");
            } else {
                $error_msg = is_array($embedding_result) && isset($embedding_result['error']) ? $embedding_result['error'] : 'Unknown error';
                Log::info("Path Pilot: Failed to get embedding for post ID $id. Error: $error_msg");
                $errors[$id] = $error_msg;
            }
            $processed++;
            sleep(1); // avoid rate limits
        }
        wp_send_json_success(['processed' => $processed, 'errors' => $errors]);
    }

    /**
     * Enqueue admin JS for embedding recompute
     */
    public function enqueue_admin_js($hook) {
        // Only load these scripts on our plugin's pages
        if (!$this->is_path_pilot_screen()) {
            return;
        }

        // For the settings page, load the main admin script with API access
        if ('path-pilot_page_path-pilot-settings' === $hook || 'toplevel_page_path-pilot' === $hook) {
            wp_enqueue_script(
                'path-pilot-admin-script',
                plugin_dir_url(__FILE__) . '../../admin/admin.js',
                ['jquery', 'wp-api'], // Add wp-api dependency for the nonce
                get_path_pilot_pro_version(),
                true
            );
        }

        // For the home/analytics page, load the embeddings script
        if ('toplevel_page_path-pilot' === $hook) {
            wp_enqueue_script(
                'path-pilot-embeddings-js',
                plugin_dir_url(__FILE__) . '../../admin/pro/embeddings.js',
                ['jquery'],
                get_path_pilot_pro_version(),
                true
            );
            wp_localize_script(
                'path-pilot-embeddings-js',
                'pp_admin_ajax',
                [
                    'ajax_url' => admin_url('admin-ajax.php'),
                    'nonce'    => wp_create_nonce('path_pilot_admin_ajax_nonce'),
                ]
            );
        }
    }

    public function render_embedding_coverage($total_pages) {
        $tuple = $this->embedding_coverage($total_pages);
        $embedding_coverage = $tuple[0];
        $pages_with_embeddings = $tuple[1];
        ?>
        <div class="pp-home-stat pp-stat-card">
            <div class="pp-home-stat-label">Embedding Coverage <span class="pp-pro-tag">PRO</span></div>
            <div id="pp-embedding-coverage-value" class="pp-home-stat-value"><?php echo $pages_with_embeddings; ?> / <?php echo $total_pages; ?></div>

            <?php if (!$this->pro_features_active()): ?>
                <!-- Pro Features Disabled -->
                <div class="pp-stat-waiting" style="color: #e67e22;">Pro features disabled</div>
                <div class="pp-progress-bar" style="opacity: 0.5;">
                    <div class="pp-progress-value" style="width: <?php echo $embedding_coverage; ?>%; background-color: #f39c12;"></div>
                </div>
                <div class="pp-stat-description" style="color: #e67e22;">
                    <strong>Semantic embeddings require a valid license key.</strong><br>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=path-pilot-settings')); ?>" style="color: #e67e22; text-decoration: underline;">Configure your license key</a> to enable AI-powered recommendations.
                </div>
            <?php else: ?>
                <!-- Single progress bar container that's always present -->
                <div id="pp-embedding-progress-container">
                    <div id="pp-embedding-status" class="pp-stat-waiting">
                        <?php if ($embedding_coverage < 100): ?>
                            Some pages are missing embeddings.
                        <?php else: ?>
                            100% coverage
                        <?php endif; ?>
                    </div>
                    <div class="pp-progress-bar">
                        <div id="pp-embedding-progress-bar" class="pp-progress-value" style="width: <?php echo $embedding_coverage; ?>%"></div>
                    </div>
                </div>

                <!-- Progress messages during sync -->
                <div id="pp-embedding-progress"></div>

                <!-- Status description for 100% coverage -->
                <?php if ($embedding_coverage >= 100): ?>
                    <div class="pp-stat-description">
                        All published pages have up-to-date semantic embeddings!
                    </div>
                <?php endif; ?>

                <!-- Buttons -->
                <div>
                    <?php if ($embedding_coverage < 100): ?>
                        <button id="pp-recompute-embeddings-ajax" class="btn btn-primary" style="padding:8px 18px;font-weight:600;font-size:1rem;margin-top:12px;">Sync</button>
                    <?php endif; ?>
                    <button id="pp-full-refresh-embeddings-ajax" class="btn btn-secondary" style="padding:8px 18px;font-weight:600;font-size:1rem;margin-top:12px;">Full Refresh</button>
                </div>
            <?php endif; ?>
        </div
        <?php
    }
}
