jQuery(document).ready(function($) {

    const recomputeBtn = $('#pp-recompute-embeddings-ajax');
    const embeddingProgressContainer = $('#pp-embedding-progress-container');
    // Use specific IDs for progress bar and status
    const embeddingProgressBar = $('#pp-embedding-progress-bar');
    const embeddingStatus = $('#pp-embedding-status');
    const embeddingSyncStatus = $('#pp-embedding-progress');

    let pagesToProcess = [];
    let currentBatchIndex = 0;
    const batchSize = 5; // Process 5 pages at a time
    var running = false; // Declare running at top level

    // Function to update the UI with current progress
    function updateProgressUI(processedCount, totalCount) {
        const percentage = totalCount > 0 ? Math.round((processedCount / totalCount) * 100) : 0;
        
        // Update progress bar using specific ID
        if (embeddingProgressBar.length > 0) {
            embeddingProgressBar.css('width', `${percentage}%`);
        }
        
        // Update the coverage value display to show current progress
        const coverageValueElement = $('#pp-embedding-coverage-value');
        if (coverageValueElement.length > 0 && totalCount > 0) {
            coverageValueElement.text(`${processedCount} / ${totalCount}`);
        }
        
        // Update status text using specific ID
        if (embeddingStatus.length > 0 && percentage < 100) {
            embeddingStatus.text(`Processing embeddings: ${percentage}% complete`);
        }
        
        // Ensure the progress container is visible during processing
        if (totalCount > 0 && percentage < 100) {
            $('#pp-embedding-progress-container').show();
        }
        
        // Update status text - make sure it's visible
        if (processedCount === 0 && totalCount > 0) {
            embeddingSyncStatus.html(`<div style="margin-bottom: 8px; font-size: 0.9rem; color: #666;">Starting refresh: 0 / ${totalCount} pages (0% complete)</div>`).show();
        } else if (percentage < 100 && totalCount > 0) {
            embeddingSyncStatus.html(`<div style="margin-bottom: 8px; font-size: 0.9rem; color: #666;">Processing embeddings: ${processedCount} / ${totalCount} pages (${percentage}% complete)</div>`).show();
        }
        
        // Update button states
        recomputeBtn.prop('disabled', true).text('Processing...');
        $('#pp-full-refresh-embeddings-ajax').prop('disabled', true).text('Processing...');

        if (percentage === 100 && totalCount > 0) {
            embeddingSyncStatus.html('<div style="margin-bottom: 8px; font-size: 0.9rem; color: #4caf50; font-weight: 600;">✅ Embeddings sync complete!</div>');
            // Update the main status text
            if (embeddingStatus.length > 0) {
                embeddingStatus.text('Embeddings sync complete!');
            }
            recomputeBtn.prop('disabled', false).text('Sync Missing');
            $('#pp-full-refresh-embeddings-ajax').prop('disabled', false).text('Full Refresh');
            running = false;
            
            // Refresh the page after a short delay to show updated stats
            setTimeout(() => {
                location.reload();
            }, 2000);
        } else if (totalCount === 0) {
            embeddingSyncStatus.html('<div style="margin-bottom: 8px; font-size: 0.9rem; color: #4caf50; font-weight: 600;">✅ No pages found needing embeddings. All up to date!</div>');
            recomputeBtn.prop('disabled', false).text('Sync Missing');
            $('#pp-full-refresh-embeddings-ajax').prop('disabled', false).text('Full Refresh');
            running = false;
        }
    }

    // Function to handle errors during the sync process
    function handleSyncError(message) {
        embeddingSyncStatus.html(`<div style="margin-bottom: 8px; font-size: 0.9rem; color: #e74c3c; font-weight: 600;">❌ Error: ${message}</div>`).show();
        recomputeBtn.prop('disabled', false).text('Sync');
    }

    // Step 1: Get all pages that need embeddings
    function fetchPagesNeedingEmbeddings() {
        embeddingSyncStatus.html('<div style="margin-bottom: 8px; font-size: 0.9rem; color: #666;">🔍 Fetching pages needing embeddings...</div>').show();
        recomputeBtn.prop('disabled', true).text('Fetching...');

        $.ajax({
            url: pp_admin_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'get_pages_needing_embeddings',
                _wpnonce: pp_admin_ajax.nonce
            },
            success: function(response) {
                if (response.success && response.data.ids) {
                    pagesToProcess = response.data.ids;
                    currentBatchIndex = 0;
                    if (pagesToProcess.length > 0) {
                        processNextBatch();
                    } else {
                        updateProgressUI(0, 0); // No pages to process
                    }
                } else {
                    handleSyncError(response.data.message || 'Failed to get page list.');
                }
            },
            error: function() {
                handleSyncError('Failed to connect to server for page list.');
            }
        });
    }

    // Step 2: Process pages in batches
    function processNextBatch() {
        if (currentBatchIndex < pagesToProcess.length) {
            const batch = pagesToProcess.slice(currentBatchIndex, currentBatchIndex + batchSize);
            
            // Show current processing status (don't update as if completed yet)
            embeddingSyncStatus.text(`Processing batch ${Math.floor(currentBatchIndex / batchSize) + 1}...`).show();

            $.ajax({
                url: pp_admin_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'recompute_embeddings_batch',
                    ids: batch,
                    _wpnonce: pp_admin_ajax.nonce
                },
                success: function(response) {
                    if (response.success) {
                        // Update index based on actual items processed
                        currentBatchIndex += response.data.processed;
                        
                        // Update progress UI with actual progress
                        updateProgressUI(currentBatchIndex, pagesToProcess.length);
                        
                        // Check if any errors in batch
                        if (Object.keys(response.data.errors).length > 0) {
                            console.warn('Errors in batch processing:', response.data.errors);
                            // You might want to display these errors to the user or log them more visibly
                        }
                        
                        // Continue processing if there are more items
                        if (currentBatchIndex < pagesToProcess.length) {
                            setTimeout(processNextBatch, 1000); // Wait 1 sec to avoid rate limits
                        } else {
                            // All processing complete
                            updateProgressUI(pagesToProcess.length, pagesToProcess.length);
                        }
                    } else {
                        handleSyncError(response.data.message || 'Failed to process batch.');
                    }
                },
                error: function() {
                    handleSyncError('Failed to connect to server for batch processing.');
                }
            });
        } else {
            updateProgressUI(pagesToProcess.length, pagesToProcess.length); // Final update
        }
    }

    // Handler for full refresh button
    var $fullBtn = $('#pp-full-refresh-embeddings-ajax');
    $fullBtn.on('click', function(e) {
        e.preventDefault();
        if (running) return;
        if (!confirm('Full refresh will regenerate embeddings for ALL pages and consume API credits accordingly. This may take time and use more credits. Are you sure?')) return;
        running = true;
        $fullBtn.prop('disabled', true).text('Starting...');
        recomputeBtn.prop('disabled', true); // Disable delta button too
        
        // Reset progress bar to 0% and show fetching status
        embeddingProgressBar.css('width', '0%');
        
        // Ensure progress elements are visible
        $('#pp-embedding-progress-container').show();
        embeddingSyncStatus.html('<div style="margin-bottom: 8px; font-size: 0.9rem; color: #666;">🔍 Fetching all pages for full refresh...</div>').show();
        
        $.ajax({
            url: pp_admin_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'get_pages_needing_embeddings',
                _wpnonce: pp_admin_ajax.nonce,
                full_refresh: true // Flag for full refresh
            },
            success: function(response) {
                if (response.success && response.data.ids) {
                    pagesToProcess = response.data.ids;
                    currentBatchIndex = 0;
                    if (pagesToProcess.length > 0) {
                        // Initialize progress display with 0 processed
                        updateProgressUI(0, pagesToProcess.length);
                        processNextBatch();
                    } else {
                        updateProgressUI(0, 0); // No pages to process
                        $fullBtn.prop('disabled', false).text('Full Refresh');
                        recomputeBtn.prop('disabled', false);
                        running = false;
                    }
                } else {
                    handleSyncError(response.data.message || 'Failed to get page list for full refresh.');
                    $fullBtn.prop('disabled', false).text('Full Refresh');
                    recomputeBtn.prop('disabled', false);
                    running = false;
                }
            },
            error: function() {
                handleSyncError('Failed to connect to server for full refresh.');
                $fullBtn.prop('disabled', false).text('Full Refresh');
                recomputeBtn.prop('disabled', false);
                running = false;
            }
        });
    });

    // Attach click listener to the sync button
    if (recomputeBtn.length) {
        recomputeBtn.on('click', function(e) {
            e.preventDefault();
            if (running) return;
            if (!confirm('Refreshing embeddings will consume API credits for each page processed. Are you sure you want to continue?')) return;
            running = true;
            recomputeBtn.prop('disabled', true);
            fetchPagesNeedingEmbeddings();
        });
    }

    // Initial state setup (optional, depends on how your admin page loads)
    // If you want to show initial progress on page load, you'd call fetchPagesNeedingEmbeddings() here
    // Or perhaps call an AJAX to get initial counts to populate the progress bar.

}); 