/**
 * Path Pilot Admin JavaScript
 */
jQuery(document).ready(function($) {
    console.log("Path Pilot Admin JS Loaded.");

    // Fetch and display credit balance on settings page
    if ($('#credit-balance-container').length) {
        console.log("Credit balance container found. Calling fetchCreditBalance().");
        fetchCreditBalance();
    }

    // Show success message when settings are saved
    if (window.location.search.includes('settings-updated=true')) {
        const notice = $('<div class="notice notice-success is-dismissible"><p>Settings saved successfully!</p></div>');
        $('.wrap h1').after(notice);

        // Auto-dismiss after 3 seconds
        setTimeout(function() {
            notice.fadeOut();
        }, 3000);
    }

    // Toggle password visibility for API key
    $('.path-pilot-admin').on('click', '.toggle-password', function(e) {
        e.preventDefault();
        const input = $($(this).data('toggle'));
        const icon = $(this).find('i');

        if (input.attr('type') === 'password') {
            input.attr('type', 'text');
            icon.removeClass('dashicons-visibility').addClass('dashicons-hidden');
            $(this).attr('title', 'Hide API Key');
        } else {
            input.attr('type', 'password');
            icon.removeClass('dashicons-hidden').addClass('dashicons-visibility');
            $(this).attr('title', 'Show API Key');
        }
    });

    // Add password toggle to API key input
    const apiKeyInput = $('#path_pilot_openai_api_key');
    if (apiKeyInput.length) {
        const toggleBtn = $('<button type="button" class="toggle-password button-link" data-toggle="#path_pilot_openai_api_key" title="Show API Key"><span class="screen-reader-text">Show API Key</span><i class="dashicons dashicons-visibility"></i></button>');
        apiKeyInput.after(toggleBtn);
        apiKeyInput.css('padding-right', '40px');
        toggleBtn.css({
            position: 'absolute',
            right: '10px',
            top: '50%',
            transform: 'translateY(-50%)',
            background: 'none',
            border: 'none',
            cursor: 'pointer'
        });
        apiKeyInput.parent().css('position', 'relative');
    }

    // Initialize tabs if present in URL
    const urlParams = new URLSearchParams(window.location.search);
    const tab = urlParams.get('tab');
    if (tab) {
        $('.nav-link').removeClass('active');
        $(`.nav-link[href*="tab=${tab}"]`).addClass('active');
    }

    // Add confirm dialog to settings reset button if it exists
    $('#reset-settings').on('click', function(e) {
        if (!confirm('Are you sure you want to reset all Path Pilot settings to defaults? This cannot be undone.')) {
            e.preventDefault();
        }
    });

    // Enhanced form validation
    $('#path-pilot-settings-form').on('submit', function(e) {
        const apiKey = $('#path_pilot_openai_api_key').val().trim();

        // Validate OpenAI API key format if entered
        if (apiKey && !apiKey.startsWith('sk-')) {
            e.preventDefault();
            alert('Please enter a valid OpenAI API key starting with "sk-"');
            $('#path_pilot_openai_api_key').focus();
            return false;
        }

        return true;
    });

    /**
     * Shows or hides purchase buttons based on subscription status
     */
    function handlePurchaseButtonsVisibility(hasActiveSubscription, subscriptionId) {
        const purchaseButtonsContainer = $('.credit-purchase-options');
        const buyMoreHeading = purchaseButtonsContainer.prev('h5'); // The "Buy More Credits" heading
        const subscriptionPrompt = $('#subscription-prompt');

        if (hasActiveSubscription && subscriptionId) {
            // Show purchase buttons for active subscribers
            purchaseButtonsContainer.show();
            buyMoreHeading.show();
            if (subscriptionPrompt.length) {
                subscriptionPrompt.hide();
            }
            console.log('Showing purchase buttons for subscription:', subscriptionId);
        } else {
            // Hide purchase buttons and show subscription prompt for non-subscribers
            purchaseButtonsContainer.hide();
            buyMoreHeading.hide();

            if (subscriptionPrompt.length === 0) {
                // Create subscription prompt if it doesn't exist
                const promptHtml = `
                    <div id="subscription-prompt" style="margin: 10px 0; padding: 12px; background: #f0f8ff; border: 1px solid #b3d9ff; border-radius: 4px;">
                        <p style="margin: 0; color: #2c3e50; font-size: 14px;">
                            <strong>💡 Need more credits?</strong>
                            <a href="https://pathpilot.app/pricing" target="_blank" style="color: #0073aa; text-decoration: none; font-weight: 600;">Subscribe to Path Pilot</a>
                            to purchase additional anytime credits that never expire.
                        </p>
                    </div>
                `;
                buyMoreHeading.after(promptHtml);
            } else {
                subscriptionPrompt.show();
            }
            console.log('Hiding purchase buttons - no active subscription');
        }
    }

    /**
     * Fetches the credit balance from the REST API and updates the display.
     */
    function fetchCreditBalance() {
        console.log("Inside fetchCreditBalance().");
        const loadingElem = $('#credit-balance-loading');
        const displayContainer = $('#credit-balance-display');
        const progressBar = $('#credit-progress');
        const textElem = $('#credit-text');
        const breakdownContainer = $('#credit-breakdown');
        const breakdownTextElem = $('#credit-breakdown-text');

        // Use the global nonce provided by WordPress for authentication
        const nonce = typeof wpApiSettings !== 'undefined' ? wpApiSettings.nonce : null;
        console.log("WP API Nonce value is:", nonce);

        if (!nonce) {
            loadingElem.text('Error: Auth Nonce not found.');
            console.error('WP API Settings or Nonce not available. Make sure wp-api is an enqueued script dependency.');
            return;
        }

        console.log("Nonce found. Proceeding with AJAX request.");
        $.ajax({
            url: '/wp-json/path-pilot/v1/credits',
            method: 'GET',
            beforeSend: function (xhr) {
                console.log("Making AJAX call to /credits endpoint.");
                xhr.setRequestHeader('X-WP-Nonce', nonce);
            },
            success: function(response) {
                console.log("Received data:", response);

                // Check if we have the new multi-key structure
                if (response && response.monthly && response.anytime) {
                    const monthlyRemaining = response.monthly.remaining;
                    const monthlyTotal = response.monthly.total;
                    const anytimeRemaining = response.anytime.remaining;
                    const anytimeTotal = response.anytime.total;
                    const totalRemaining = response.total_remaining;
                    const totalCredits = response.total_credits;
                    const hasActiveSubscription = response.has_active_subscription;
                    const subscriptionId = response.subscription_id;

                    const percentage = totalCredits > 0 ? (totalRemaining / totalCredits) * 100 : 0;

                    // Format numbers
                    const formattedMonthlyRemaining = monthlyRemaining.toLocaleString();
                    const formattedMonthlyTotal = monthlyTotal.toLocaleString();
                    const formattedAnytimeRemaining = anytimeRemaining.toLocaleString();
                    const formattedAnytimeTotal = anytimeTotal.toLocaleString();
                    const formattedTotalRemaining = totalRemaining.toLocaleString();
                    const formattedTotalCredits = totalCredits.toLocaleString();

                    // Hide loading and show display
                    loadingElem.hide();
                    displayContainer.show();

                    // Build credit text and breakdown
                    let creditText;
                    if (anytimeTotal > 0) {
                        // Show total when anytime credits exist
                        creditText = `Total: ${formattedTotalRemaining} / ${formattedTotalCredits} Credits Remaining`;

                        // Show breakdown with tooltip
                        breakdownTextElem.text(`Monthly: ${formattedMonthlyRemaining} / ${formattedMonthlyTotal} | Anytime: ${formattedAnytimeRemaining} / ${formattedAnytimeTotal}`);
                        breakdownContainer.show();

                        // Update tooltip for anytime credits
                        const helpTextElem = $('#credit-help-text');
                        helpTextElem.html(`
                            <strong>Monthly:</strong> Subscription credits, reset monthly<br>
                            <strong>Anytime:</strong> Purchased credits, never expire
                        `);
                    } else {
                        // Only show monthly credits when no anytime credits exist
                        creditText = `${formattedMonthlyRemaining} / ${formattedMonthlyTotal} Credits Remaining`;
                        breakdownContainer.hide();
                    }

                    // Update text and progress bar
                    textElem.html(creditText);
                    progressBar.css('width', percentage + '%');

                    // Change progress bar color based on usage
                    if (percentage < 10) {
                        progressBar.css('background-color', '#d63638'); // Red
                    } else if (percentage < 25) {
                        progressBar.css('background-color', '#f9a825'); // Yellow
                    } else {
                        progressBar.css('background-color', '#2ecc40'); // Green
                    }

                    // Show/hide purchase buttons based on subscription status
                    handlePurchaseButtonsVisibility(hasActiveSubscription, subscriptionId);

                } else if (response && typeof response.remaining !== 'undefined' && typeof response.total !== 'undefined') {
                    // Fallback for old single-key structure
                    const remaining = response.remaining;
                    const total = response.total;
                    const percentage = total > 0 ? (remaining / total) * 100 : 0;

                    const formattedRemaining = remaining.toLocaleString();
                    const formattedTotal = total.toLocaleString();

                    // Hide loading and show display
                    loadingElem.hide();
                    displayContainer.show();

                    // Update text and progress bar
                    textElem.text(`${formattedRemaining} / ${formattedTotal} Credits Remaining`);
                    progressBar.css('width', percentage + '%');
                    breakdownContainer.hide();

                    // Change progress bar color based on usage
                    if (percentage < 10) {
                        progressBar.css('background-color', '#d63638'); // Red
                    } else if (percentage < 25) {
                        progressBar.css('background-color', '#f9a825'); // Yellow
                    } else {
                        progressBar.css('background-color', '#2ecc40'); // Green
                    }

                    // For old structure, assume no active subscription
                    handlePurchaseButtonsVisibility(false, '');

                } else {
                    loadingElem.hide();
                    displayContainer.show();
                    textElem.text('N/A');
                    progressBar.css('width', '0%');
                    breakdownContainer.hide();

                    // Hide purchase buttons when no valid data
                    handlePurchaseButtonsVisibility(false, '');
                }
            },
            error: function(jqXHR) {
                let errorMessage = 'Error loading credits';
                if (jqXHR.responseJSON && jqXHR.responseJSON.error) {
                    errorMessage = jqXHR.responseJSON.error;
                } else if (jqXHR.status === 403) {
                    errorMessage = 'Permission Denied';
                }

                loadingElem.hide();
                displayContainer.show();
                textElem.text(errorMessage).css({
                    'color': '#d63638'
                });
                breakdownContainer.hide();

                // Hide purchase buttons on error (no subscription status available)
                handlePurchaseButtonsVisibility(false, '');
            }
        });
    }

    /**
     * Tests an API key by attempting to fetch credit balance
     */
    function testApiKey(apiKey, resultContainer) {
        if (!apiKey || apiKey.trim() === '') {
            showTestResult(resultContainer, false, 'Please enter an API key first.');
            return;
        }

        // Show loading state
        showTestResult(resultContainer, null, 'Testing API key connection...');

        const nonce = typeof wpApiSettings !== 'undefined' ? wpApiSettings.nonce : null;
        if (!nonce) {
            showTestResult(resultContainer, false, 'Authentication error: No nonce available.');
            return;
        }

        $.ajax({
            url: '/wp-json/path-pilot/v1/test-api-key',
            method: 'POST',
            contentType: 'application/json',
            data: JSON.stringify({
                api_key: apiKey
            }),
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-WP-Nonce', nonce);
            },
            success: function(response) {
                if (response.valid) {
                    const credits = response.credits;
                    const subscription = response.subscription;

                    let message = 'API key is valid and working! ';
                    if (credits && credits.total > 0) {
                        message += `You have ${credits.remaining.toLocaleString()} of ${credits.total.toLocaleString()} credits remaining.`;
                    }
                    if (subscription && subscription.has_active_subscription) {
                        message += ' Active subscription detected.';
                    }

                    showTestResult(resultContainer, true, message);
                } else {
                    showTestResult(resultContainer, false, response.error || 'API key validation failed.');
                }
            },
            error: function(jqXHR) {
                let errorMessage = 'API key test failed';
                if (jqXHR.responseJSON && jqXHR.responseJSON.error) {
                    errorMessage = jqXHR.responseJSON.error;
                } else if (jqXHR.status === 400) {
                    errorMessage = 'Invalid request - please check your API key format';
                } else if (jqXHR.status === 403) {
                    errorMessage = 'Permission denied - please refresh the page and try again';
                } else if (jqXHR.status === 500) {
                    errorMessage = 'Server error - please try again later';
                }

                showTestResult(resultContainer, false, errorMessage);
            }
        });
    }

    /**
     * Shows the result of API key testing
     */
    function showTestResult(container, success, message) {
        container.show();

        if (success === null) {
            // Loading state
            container.removeClass('success error').addClass('loading');
            container.css({
                'background': '#f0f8ff',
                'border': '1px solid #b3d9ff',
                'color': '#2c3e50'
            });
            container.html('<strong>🔄 Testing...</strong> ' + message);
        } else if (success) {
            // Success state
            container.removeClass('loading error').addClass('success');
            container.css({
                'background': '#eafaf1',
                'border': '1px solid #2ecc40',
                'color': '#27ae60'
            });
            container.html('<strong>✅ Success!</strong> ' + message);
        } else {
            // Error state
            container.removeClass('loading success').addClass('error');
            container.css({
                'background': '#ffeaea',
                'border': '1px solid #ff6b6b',
                'color': '#d63031'
            });
            container.html('<strong>❌ Error:</strong> ' + message);
        }
    }

    // Handle API key testing button
    $('#test-api-key').on('click', function(e) {
        e.preventDefault();
        const apiKey = $('#path_pilot_api_key').val().trim();
        const resultContainer = $('#api-test-result');

        testApiKey(apiKey, resultContainer);
    });

    // Auto-test API key when field loses focus (if it has content)
    $('#path_pilot_api_key').on('blur', function() {
        const apiKey = $(this).val().trim();
        const resultContainer = $('#api-test-result');

        // Only auto-test if there's content and we're in the pre-subscription state
        if (apiKey && $('#path-pilot-api-key-form').length > 0) {
            // Small delay to avoid testing while user is still typing
            setTimeout(function() {
                testApiKey(apiKey, resultContainer);
            }, 500);
        }
    });

    // Store original API key value for comparison
    $('#path_pilot_api_key').each(function() {
        $(this).data('original-value', $(this).val());
    });
});
